"use strict";
/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.dismissDialog = exports.dangerDialog = exports.acceptDialog = exports.waitForDialog = exports.framePromise = exports.isFulfilled = exports.signalToPromise = exports.signalToPromises = exports.doLater = exports.expectFailure = exports.testEmission = exports.simulate = exports.sleep = void 0;
const simulate_event_1 = require("simulate-event");
const coreutils_1 = require("@lumino/coreutils");
const signaling_1 = require("@lumino/signaling");
const testutils_1 = require("@jupyterlab/coreutils/lib/testutils");
var testutils_2 = require("@jupyterlab/coreutils/lib/testutils");
Object.defineProperty(exports, "sleep", { enumerable: true, get: function () { return testutils_2.sleep; } });
// Add a simple polyfill for `PointerEvent` which is not yet supported by jsdom
// see https://github.com/jsdom/jsdom/pull/2666
if (!global.PointerEvent) {
    class PointerEvent extends MouseEvent {
    }
    global.PointerEvent = PointerEvent;
}
const POINTER_EVENTS = [
    'pointerdown',
    'pointerenter',
    'pointerleave',
    'pointermove',
    'pointerout',
    'pointerover',
    'pointerup'
];
/**
 * Extends `simulate` from no longer actively developed `simulate-event`
 * with a subset of `pointer` events.
 */
function simulate(element, type, options) {
    if (POINTER_EVENTS.includes(type)) {
        element.dispatchEvent(new PointerEvent(type, options));
    }
    else {
        (0, simulate_event_1.simulate)(element, type, options);
    }
}
exports.simulate = simulate;
/**
 * Test a single emission from a signal.
 *
 * @param signal - The signal we are listening to.
 * @param find - An optional function to determine which emission to test,
 * defaulting to the first emission.
 * @param test - An optional function which contains the tests for the emission, and should throw an error if the tests fail.
 * @param value - An optional value that the promise resolves to if the test is
 * successful.
 *
 * @returns a promise that rejects if the function throws an error (e.g., if an
 * expect test doesn't pass), and resolves otherwise.
 *
 * #### Notes
 * The first emission for which the find function returns true will be tested in
 * the test function. If the find function is not given, the first signal
 * emission will be tested.
 *
 * You can test to see if any signal comes which matches a criteria by just
 * giving a find function. You can test the very first signal by just giving a
 * test function. And you can test the first signal matching the find criteria
 * by giving both.
 *
 * The reason this function is asynchronous is so that the thing causing the
 * signal emission (such as a websocket message) can be asynchronous.
 */
async function testEmission(signal, options = {}) {
    const done = new coreutils_1.PromiseDelegate();
    const object = {};
    signal.connect((sender, args) => {
        var _a, _b, _c;
        if ((_b = (_a = options.find) === null || _a === void 0 ? void 0 : _a.call(options, sender, args)) !== null && _b !== void 0 ? _b : true) {
            try {
                signaling_1.Signal.disconnectReceiver(object);
                if (options.test) {
                    options.test(sender, args);
                }
            }
            catch (e) {
                done.reject(e);
            }
            done.resolve((_c = options.value) !== null && _c !== void 0 ? _c : undefined);
        }
    }, object);
    return done.promise;
}
exports.testEmission = testEmission;
/**
 * Expect a failure on a promise with the given message.
 */
async function expectFailure(promise, message) {
    let called = false;
    try {
        await promise;
        called = true;
    }
    catch (err) {
        if (message && err.message.indexOf(message) === -1) {
            throw Error(`Error "${message}" not in: "${err.message}"`);
        }
    }
    if (called) {
        throw Error(`Failure was not triggered, message was: ${message}`);
    }
}
exports.expectFailure = expectFailure;
/**
 * Do something in the future ensuring total ordering with respect to promises.
 */
async function doLater(cb) {
    await Promise.resolve(void 0);
    cb();
}
exports.doLater = doLater;
/**
 * Convert a signal into an array of promises.
 *
 * @param signal - The signal we are listening to.
 * @param numberValues - The number of values to store.
 *
 * @returns a Promise that resolves with an array of `(sender, args)` pairs.
 */
function signalToPromises(signal, numberValues) {
    const values = new Array(numberValues);
    const resolvers = new Array(numberValues);
    for (let i = 0; i < numberValues; i++) {
        values[i] = new Promise(resolve => {
            resolvers[i] = resolve;
        });
    }
    let current = 0;
    function slot(sender, args) {
        resolvers[current++]([sender, args]);
        if (current === numberValues) {
            cleanup();
        }
    }
    signal.connect(slot);
    function cleanup() {
        signal.disconnect(slot);
    }
    return values;
}
exports.signalToPromises = signalToPromises;
/**
 * Convert a signal into a promise for the first emitted value.
 *
 * @param signal - The signal we are listening to.
 *
 * @returns a Promise that resolves with a `(sender, args)` pair.
 */
function signalToPromise(signal) {
    return signalToPromises(signal, 1)[0];
}
exports.signalToPromise = signalToPromise;
/**
 * Test to see if a promise is fulfilled.
 *
 * @param delay - optional delay in milliseconds before checking
 * @returns true if the promise is fulfilled (either resolved or rejected), and
 * false if the promise is still pending.
 */
async function isFulfilled(p, delay = 0) {
    const x = Object.create(null);
    let race;
    if (delay > 0) {
        race = (0, testutils_1.sleep)(delay, x);
    }
    else {
        race = x;
    }
    const result = await Promise.race([p, race]).catch(() => false);
    return result !== x;
}
exports.isFulfilled = isFulfilled;
/**
 * Convert a requestAnimationFrame into a Promise.
 */
function framePromise() {
    const done = new coreutils_1.PromiseDelegate();
    requestAnimationFrame(() => {
        done.resolve(void 0);
    });
    return done.promise;
}
exports.framePromise = framePromise;
/**
 * Wait for a dialog to be attached to an element.
 */
async function waitForDialog(host = document.body, timeout = 250) {
    const interval = 25;
    const limit = Math.floor(timeout / interval);
    for (let counter = 0; counter < limit; counter++) {
        if (host.getElementsByClassName('jp-Dialog')[0]) {
            return;
        }
        await (0, testutils_1.sleep)(interval);
    }
    throw new Error('Dialog not found');
}
exports.waitForDialog = waitForDialog;
/**
 * Accept a dialog after it is attached by accepting the default button.
 */
async function acceptDialog(host = document.body, timeout = 250) {
    await waitForDialog(host, timeout);
    const node = host.getElementsByClassName('jp-Dialog')[0];
    if (node) {
        simulate(node, 'keydown', { keyCode: 13 });
    }
}
exports.acceptDialog = acceptDialog;
/**
 * Click on the warning button in a dialog after it is attached
 */
async function dangerDialog(host = document.body, timeout = 250) {
    await waitForDialog(host, timeout);
    const node = host.getElementsByClassName('jp-mod-warn')[0];
    if (node) {
        simulate(node, 'click', { button: 1 });
    }
}
exports.dangerDialog = dangerDialog;
/**
 * Dismiss a dialog after it is attached.
 *
 * #### Notes
 * This promise will always resolve successfully.
 */
async function dismissDialog(host = document.body, timeout = 250) {
    try {
        await waitForDialog(host, timeout);
    }
    catch (error) {
        return; // Ignore calls to dismiss the dialog if there is no dialog.
    }
    const node = host.getElementsByClassName('jp-Dialog')[0];
    if (node) {
        simulate(node, 'keydown', { keyCode: 27 });
    }
}
exports.dismissDialog = dismissDialog;
//# sourceMappingURL=common.js.map