"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.NbConvertManager = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const serverconnection_1 = require("../serverconnection");
const coreutils_2 = require("@lumino/coreutils");
/**
 * The url for the lab nbconvert service.
 */
const NBCONVERT_SETTINGS_URL = 'api/nbconvert';
/**
 * The nbconvert API service manager.
 */
class NbConvertManager {
    /**
     * Create a new nbconvert manager.
     */
    constructor(options = {}) {
        var _a;
        this._exportFormats = null;
        this.serverSettings =
            (_a = options.serverSettings) !== null && _a !== void 0 ? _a : serverconnection_1.ServerConnection.makeSettings();
    }
    /**
     * Fetch and cache the export formats from the expensive nbconvert handler.
     */
    async fetchExportFormats() {
        this._requestingFormats = new coreutils_2.PromiseDelegate();
        this._exportFormats = null;
        const base = this.serverSettings.baseUrl;
        const url = coreutils_1.URLExt.join(base, NBCONVERT_SETTINGS_URL);
        const { serverSettings } = this;
        const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
            throw err;
        }
        const data = await response.json();
        const exportList = {};
        const keys = Object.keys(data);
        keys.forEach(function (key) {
            const mimeType = data[key].output_mimetype;
            exportList[key] = { output_mimetype: mimeType };
        });
        this._exportFormats = exportList;
        this._requestingFormats.resolve(exportList);
        return exportList;
    }
    /**
     * Get the list of export formats, preferring pre-cached ones.
     */
    async getExportFormats(force = true) {
        if (this._requestingFormats) {
            return this._requestingFormats.promise;
        }
        if (force || !this._exportFormats) {
            return await this.fetchExportFormats();
        }
        return this._exportFormats;
    }
}
exports.NbConvertManager = NbConvertManager;
//# sourceMappingURL=index.js.map