// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { VDomModel } from '@jupyterlab/ui-components';
import { Debouncer } from '@lumino/polling';
import { Signal } from '@lumino/signaling';
/**
 * Search in a document model.
 */
export class SearchDocumentModel extends VDomModel {
    /**
     * Search document model
     * @param searchProvider Provider for the current document
     * @param searchDebounceTime Debounce search time
     */
    constructor(searchProvider, searchDebounceTime) {
        super();
        this.searchProvider = searchProvider;
        this._caseSensitive = false;
        this._disposed = new Signal(this);
        this._parsingError = '';
        this._preserveCase = false;
        this._initialQuery = '';
        this._filters = {};
        this._replaceText = '';
        this._searchActive = false;
        this._searchExpression = '';
        this._useRegex = false;
        this._wholeWords = false;
        this._filters = {};
        if (this.searchProvider.getFilters) {
            const filters = this.searchProvider.getFilters();
            for (const filter in filters) {
                this._filters[filter] = filters[filter].default;
            }
        }
        searchProvider.stateChanged.connect(this._onProviderStateChanged, this);
        this._searchDebouncer = new Debouncer(() => {
            this._updateSearch().catch(reason => {
                console.error('Failed to update search on document.', reason);
            });
        }, searchDebounceTime);
    }
    /**
     * Whether the search is case sensitive or not.
     */
    get caseSensitive() {
        return this._caseSensitive;
    }
    set caseSensitive(v) {
        if (this._caseSensitive !== v) {
            this._caseSensitive = v;
            this.stateChanged.emit();
            this.refresh();
        }
    }
    /**
     * Current highlighted match index.
     */
    get currentIndex() {
        return this.searchProvider.currentMatchIndex;
    }
    /**
     * A signal emitted when the object is disposed.
     */
    get disposed() {
        return this._disposed;
    }
    /**
     * Filter values.
     */
    get filters() {
        return this._filters;
    }
    /**
     * Filter definitions for the current provider.
     */
    get filtersDefinition() {
        var _a, _b, _c;
        return (_c = (_b = (_a = this.searchProvider).getFilters) === null || _b === void 0 ? void 0 : _b.call(_a)) !== null && _c !== void 0 ? _c : {};
    }
    /**
     * Filter definitions changed.
     */
    get filtersDefinitionChanged() {
        return this.searchProvider.filtersChanged || null;
    }
    /**
     * The initial query string.
     */
    get initialQuery() {
        return this._initialQuery;
    }
    set initialQuery(v) {
        if (v) {
            // Usually the value comes from user selection (set by search provider).
            this._initialQuery = v;
        }
        else {
            // If user selection is empty, we fall back to most recent value (if any).
            this._initialQuery = this._searchExpression;
        }
    }
    /**
     * Initial query as suggested by provider.
     *
     * A common choice is the text currently selected by the user.
     */
    get suggestedInitialQuery() {
        return this.searchProvider.getInitialQuery();
    }
    /**
     * Whether the document is read-only or not.
     */
    get isReadOnly() {
        return this.searchProvider.isReadOnly;
    }
    /**
     * Replace options support.
     */
    get replaceOptionsSupport() {
        return this.searchProvider.replaceOptionsSupport;
    }
    /**
     * Parsing regular expression error message.
     */
    get parsingError() {
        return this._parsingError;
    }
    /**
     * Whether to preserve case when replacing.
     */
    get preserveCase() {
        return this._preserveCase;
    }
    set preserveCase(v) {
        if (this._preserveCase !== v) {
            this._preserveCase = v;
            this.stateChanged.emit();
            this.refresh();
        }
    }
    /**
     * Replacement expression
     */
    get replaceText() {
        return this._replaceText;
    }
    set replaceText(v) {
        if (this._replaceText !== v) {
            this._replaceText = v;
            this.stateChanged.emit();
        }
    }
    /**
     * Search expression
     */
    get searchExpression() {
        return this._searchExpression;
    }
    set searchExpression(v) {
        if (this._searchExpression !== v) {
            this._searchExpression = v;
            this.stateChanged.emit();
            this.refresh();
        }
    }
    /**
     * Total number of matches.
     */
    get totalMatches() {
        return this.searchProvider.matchesCount;
    }
    /**
     * Whether to use regular expression or not.
     */
    get useRegex() {
        return this._useRegex;
    }
    set useRegex(v) {
        if (this._useRegex !== v) {
            this._useRegex = v;
            this.stateChanged.emit();
            this.refresh();
        }
    }
    /**
     * Whether to match whole words or not.
     */
    get wholeWords() {
        return this._wholeWords;
    }
    set wholeWords(v) {
        if (this._wholeWords !== v) {
            this._wholeWords = v;
            this.stateChanged.emit();
            this.refresh();
        }
    }
    /**
     * Dispose the model.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        if (this._searchExpression) {
            this.endQuery().catch(reason => {
                console.error(`Failed to end query '${this._searchExpression}.`, reason);
            });
        }
        this.searchProvider.stateChanged.disconnect(this._onProviderStateChanged, this);
        this._searchDebouncer.dispose();
        super.dispose();
    }
    /**
     * End the query.
     */
    async endQuery() {
        this._searchActive = false;
        await this.searchProvider.endQuery();
        this.stateChanged.emit();
    }
    /**
     * Highlight the next match.
     */
    async highlightNext() {
        await this.searchProvider.highlightNext();
        // Emit state change as the index needs to be updated
        this.stateChanged.emit();
    }
    /**
     * Highlight the previous match
     */
    async highlightPrevious() {
        await this.searchProvider.highlightPrevious();
        // Emit state change as the index needs to be updated
        this.stateChanged.emit();
    }
    /**
     * Refresh search
     */
    refresh() {
        this._searchDebouncer.invoke().catch(reason => {
            console.error('Failed to invoke search document debouncer.', reason);
        });
    }
    /**
     * Replace all matches.
     */
    async replaceAllMatches() {
        await this.searchProvider.replaceAllMatches(this._replaceText, {
            preserveCase: this.preserveCase,
            regularExpression: this.useRegex
        });
        // Emit state change as the index needs to be updated
        this.stateChanged.emit();
    }
    /**
     * Replace the current match.
     */
    async replaceCurrentMatch() {
        await this.searchProvider.replaceCurrentMatch(this._replaceText, true, {
            preserveCase: this.preserveCase,
            regularExpression: this.useRegex
        });
        // Emit state change as the index needs to be updated
        this.stateChanged.emit();
    }
    /**
     * Set the value of a given filter.
     *
     * @param name Filter name
     * @param v Filter value
     */
    async setFilter(name, v) {
        if (this._filters[name] !== v) {
            if (this.searchProvider.validateFilter) {
                this._filters[name] = await this.searchProvider.validateFilter(name, v);
                // If the value was changed
                if (this._filters[name] === v) {
                    this.stateChanged.emit();
                    this.refresh();
                }
            }
            else {
                this._filters[name] = v;
                this.stateChanged.emit();
                this.refresh();
            }
        }
    }
    async _updateSearch() {
        if (this._parsingError) {
            this._parsingError = '';
            this.stateChanged.emit();
        }
        try {
            const query = this.searchExpression
                ? Private.parseQuery(this.searchExpression, this.caseSensitive, this.useRegex, this.wholeWords)
                : null;
            if (query) {
                this._searchActive = true;
                await this.searchProvider.startQuery(query, this._filters);
                // Emit state change as the index needs to be updated
                this.stateChanged.emit();
            }
        }
        catch (reason) {
            this._parsingError = reason.toString();
            this.stateChanged.emit();
            console.error(`Failed to parse expression ${this.searchExpression}`, reason);
        }
    }
    _onProviderStateChanged() {
        if (this._searchActive) {
            this.refresh();
        }
    }
}
var Private;
(function (Private) {
    /**
     * Build the regular expression to use for searching.
     *
     * @param queryString Query string
     * @param caseSensitive Whether the search is case sensitive or not
     * @param regex Whether the expression is a regular expression
     * @returns The regular expression to use
     */
    function parseQuery(queryString, caseSensitive, regex, wholeWords) {
        const flag = caseSensitive ? 'gm' : 'gim';
        // escape regex characters in query if its a string search
        let queryText = regex
            ? queryString
            : queryString.replace(/[-[\]/{}()*+?.\\^$|]/g, '\\$&');
        if (wholeWords) {
            queryText = '\\b' + queryText + '\\b';
        }
        const ret = new RegExp(queryText, flag);
        // If the empty string is hit, the search logic will freeze the browser tab
        //  Trying /^/ or /$/ on the codemirror search demo, does not find anything.
        //  So this is a limitation of the editor.
        if (ret.test('')) {
            return null;
        }
        return ret;
    }
    Private.parseQuery = parseQuery;
})(Private || (Private = {}));
//# sourceMappingURL=searchmodel.js.map