--[[
    Simple script to program DIY kyber crystals
    works on real kyber crystals and EM4305 2.12x12mm chips
    simply run the program and select a profile via a number

    issues
    if you are getting errors when trying to read or write a chip
    run the cmd "lf tune" with no chip on the device, then move the chip over the coils till you see the lowest voltage. try different angles and in the center and or the edge of the antenna ring.
    once you find the lowest voltage then try running the script again.
    if thats still not working run "lf tune" again and put the chip in the best position like before
    the total voltage may be too high to reduce it slowly lower tin foil over the antenna and watch the voltage.
    the foil should be a bit bigger than the coil exact size does not matter.

    data pulled from here
    https://docs.google.com/spreadsheets/d/13P_GE6tNYpGvoVUTEQvA3SQzMqpZ-SoiWaTNoJoTV9Q/edit?usp=sharing
--]]

local cmds = require('commands')
local utils = require('utils')


function send_command(cmd)
    core.console(cmd)
    return ""
end

function get_profile_data(profile_name)
    local profiles = {
        ["wipe chip"] = {
            [0] = "00000000",
            [1] = "00000000",
            [2] = "00000000",
            [3] = "00000000",
            [4] = "00000000",
            [5] = "00000000",
            [6] = "00000000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Qui-Gon Jinn"] = {
            [0] = "00040072",
            [1] = "6147FBB3",
            [2] = "00000000",
            [3] = "000064FC",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "0C803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Qui-Gon Jinn 2"] = {
            [0] = "00040072",
            [1] = "6148C1EF",
            [2] = "00000000",
            [3] = "000050C2",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "0C803000",
            [7] = "00000000",
            [8] = "10000040",
            [9] = "00000000"
        },
        ["Yoda"] = {
            [0] = "00040072",
            [1] = "660A50D6",
            [2] = "00000000",
            [3] = "0000379F",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "00C03000",
            [7] = "00000000",
            [8] = "00100040",
            [9] = "00000000"
        },
        ["Yoda 2"] = {
            [0] = "00040072",
            [1] = "667B2FEE",
            [2] = "00000000",
            [3] = "0000BDB6",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "00C03000",
            [7] = "00000000",
            [8] = "00100040",
            [9] = "00000000"
        },
        ["Yoda 8-Ball"] = {
            [0] = "00040072",
            [1] = "67AD7FC8",
            [2] = "00000000",
            [3] = "0000E0FE",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "5D183000",
            [7] = "00000000",
            [8] = "00000140",
            [9] = "00000000"
        },
        ["Old Obi-Wan"] = {
            [0] = "00040072",
            [1] = "6147BBB9",
            [2] = "00000000",
            [3] = "0000BE24",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Old Luke"] = {
            [0] = "00040072",
            [1] = "614097AE",
            [2] = "00000000",
            [3] = "0000C134",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "25C03000",
            [7] = "00000000",
            [8] = "00100060",
            [9] = "00000000"
        },
        ["Old Obi-Wan 2"] = {
            [0] = "00040072",
            [1] = "614009A2",
            [2] = "00000000",
            [3] = "0000CF62",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "01000060",
            [9] = "00000000"
        },
        ["Old Luke 2"] = {
            [0] = "00040072",
            [1] = "75952DE5",
            [2] = "00000000",
            [3] = "00009988",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "25C03000",
            [7] = "00000000",
            [8] = "00010060",
            [9] = "00000000"
        },
        ["Old Obi-Wan 3"] = {
            [0] = "00040072",
            [1] = "65413B42",
            [2] = "00000000",
            [3] = "00001702",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "01000060",
            [9] = "00000000"
        },
        ["Mace Windu"] = {
            [0] = "00040072",
            [1] = "6147CCD4",
            [2] = "00000000",
            [3] = "0000A092",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "63C03000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Mace Windu 2"] = {
            [0] = "00040072",
            [1] = "6609B150",
            [2] = "00000000",
            [3] = "0000287E",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "63C03000",
            [7] = "00000000",
            [8] = "00010070",
            [9] = "00000000"
        },
        ["Mace Windu 3"] = {
            [0] = "00040072",
            [1] = "613F42AD",
            [2] = "00000000",
            [3] = "00002147",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "01000070",
            [9] = "00000000"
        },
        ["Mace Windu 4"] = {
            [0] = "00040072",
            [1] = "667B5B82",
            [2] = "00000000",
            [3] = "000050DF",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "10000070",
            [9] = "00000000"
        },
        ["Mace Windu 5"] = {
            [0] = "00040072",
            [1] = "614869C4",
            [2] = "00000000",
            [3] = "0000D691",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "01000070",
            [9] = "00000000"
        },
        ["Mace Windu 6"] = {
            [0] = "00040072",
            [1] = "759BEA43",
            [2] = "00000000",
            [3] = "00006CA0",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "63C03000",
            [7] = "00000000",
            [8] = "00100070",
            [9] = "00000000"
        },
        ["Mace Windu 7"] = {
            [0] = "00040072",
            [1] = "768E0D9D",
            [2] = "00000000",
            [3] = "0000668C",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "01000070",
            [9] = "00000000"
        },
        ["Temple Guard"] = {
            [0] = "00040072",
            [1] = "60954FDA",
            [2] = "00000000",
            [3] = "0000905A",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "7B003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Maz Kanata"] = {
            [0] = "00040072",
            [1] = "6679DFF4",
            [2] = "00000000",
            [3] = "0000D691",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "77403000",
            [7] = "00000000",
            [8] = "00100030",
            [9] = "00000000"
        },
        ["Maz Kanata 2"] = {
            [0] = "00040072",
            [1] = "60953999",
            [2] = "00000000",
            [3] = "0000F521",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "77403000",
            [7] = "00000000",
            [8] = "00100030",
            [9] = "00000000"
        },
        ["Temple Guard 2"] = {
            [0] = "00040072",
            [1] = "667A67C5",
            [2] = "00000000",
            [3] = "00002B9C",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "7B003000",
            [7] = "00000000",
            [8] = "10000030",
            [9] = "00000000"
        },
        ["Maz Kanata 3"] = {
            [0] = "00040072",
            [1] = "7A213721",
            [2] = "00000000",
            [3] = "000083AB",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "77403000",
            [7] = "00000000",
            [8] = "00010030",
            [9] = "00000000"
        },
        ["Chirrut Îmwe"] = {
            [0] = "00040072",
            [1] = "6094F399",
            [2] = "00000000",
            [3] = "00009519",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "14403000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Chirrut Îmwe 2"] = {
            [0] = "00040072",
            [1] = "667A9AB7",
            [2] = "00000000",
            [3] = "00003BE4",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "14403000",
            [7] = "00000000",
            [8] = "00010000",
            [9] = "00000000"
        },
        ["Ahsoka Tano"] = {
            [0] = "00040072",
            [1] = "667B1626",
            [2] = "00000000",
            [3] = "00007907",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "18003000",
            [7] = "00000000",
            [8] = "10000000",
            [9] = "00000000"
        },
        ["Chirrut Îmwe 3"] = {
            [0] = "00040072",
            [1] = "667B7E07",
            [2] = "00000000",
            [3] = "00002960",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "14403000",
            [7] = "00000000",
            [8] = "00100000",
            [9] = "00000000"
        },
        ["Darth Vader"] = {
            [0] = "00040072",
            [1] = "6148C4F8",
            [2] = "00000000",
            [3] = "0000FDFF",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Darth Sidious"] = {
            [0] = "00040072",
            [1] = "613F8964",
            [2] = "00000000",
            [3] = "0000C0C1",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "52403000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Darth Maul"] = {
            [0] = "00040072",
            [1] = "613FD2A9",
            [2] = "00000000",
            [3] = "0000DAD2",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "46C03000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "00000000"
        },
        ["Count Dooku"] = {
            [0] = "00040072",
            [1] = "6140880C",
            [2] = "00000000",
            [3] = "0000952D",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "31403000",
            [7] = "00000000",
            [8] = "00010010",
            [9] = "00000000"
        },
        ["Darth Vader 2"] = {
            [0] = "00040072",
            [1] = "667B33DC",
            [2] = "00000000",
            [3] = "0000E804",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "01000010",
            [9] = "00000000"
        },
        ["Darth Maul 2"] = {
            [0] = "00040072",
            [1] = "667B26E9",
            [2] = "00000000",
            [3] = "00007689",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "46C03000",
            [7] = "00000000",
            [8] = "00100010",
            [9] = "00000000"
        },
        ["Vader 8-Ball"] = {
            [0] = "00040072",
            [1] = "6A92B478",
            [2] = "00000000",
            [3] = "00004CD1",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "3E183000",
            [7] = "00000000",
            [8] = "00000110",
            [9] = "00000000"
        },
        ["Darth Maul 3"] = {
            [0] = "00040072",
            [1] = "7597EF7E",
            [2] = "00000000",
            [3] = "00003BC0",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "46C03000",
            [7] = "00000000",
            [8] = "00100010",
            [9] = "00000000"
        },
        ["Darth Sidious 2"] = {
            [0] = "00040072",
            [1] = "768E4402",
            [2] = "00000000",
            [3] = "0000A0D2",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "52403000",
            [7] = "00000000",
            [8] = "10000010",
            [9] = "00000000"
        },
        ["Snoke"] = {
            [0] = "00040072",
            [1] = "6540BD8F",
            [2] = "00000000",
            [3] = "000064B9",
            [4] = "0001805F",
            [5] = "000001FF",
            [6] = "1B183000",
            [7] = "00000000",
            [8] = "00001010",
            [9] = "00000000"
        },
        ["Luke Skywalker"] = {
            [0] = "00040072",
            [1] = "804B08F0",
            [2] = "00000000",
            [3] = "00006BF1",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "0C803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "050D0000"
        },
        ["Luminara Unduli"] = {
            [0] = "00040072",
            [1] = "7B83C85A",
            [2] = "00000000",
            [3] = "000052CE",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "0C803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "180D0000"
        },
        ["Plo Koon"] = {
            [0] = "00040072",
            [1] = "7B8998F3",
            [2] = "00000000",
            [3] = "00007703",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "040D0000"
        },
        ["Plo Koon 2"] = {
            [0] = "00040072",
            [1] = "7B8413EA",
            [2] = "00000000",
            [3] = "0000D8F3",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "040D0000"
        },
        ["Plo Koon 3"] = {
            [0] = "00040072",
            [1] = "7B84222B",
            [2] = "00000000",
            [3] = "000023E3",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "29803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "040D0000"
        },
        ["Mace Windu S2"] = {
            [0] = "00040072",
            [1] = "7B8936EA",
            [2] = "00000000",
            [3] = "0000520D",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "070D0000"
        },
        ["General Grievous"] = {
            [0] = "00040072",
            [1] = "7B896284",
            [2] = "00000000",
            [3] = "00008529",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "6F803000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "060D0000"
        },
        ["Rey Skywalker"] = {
            [0] = "00040072",
            [1] = "7B88F3F4",
            [2] = "00000000",
            [3] = "00001511",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "7B003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "170D0000"
        },
        ["Rey Skywalker 2"] = {
            [0] = "00040072",
            [1] = "7B841039",
            [2] = "00000000",
            [3] = "0000EA22",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "7B003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "170D0000"
        },
        ["Krin Dagbard"] = {
            [0] = "00040072",
            [1] = "7B894F46",
            [2] = "00000000",
            [3] = "00007BC2",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "18003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "140D0000"
        },
        ["Krin Dagbard 2"] = {
            [0] = "00040072",
            [1] = "7B841797",
            [2] = "00000000",
            [3] = "00006A58",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "18003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "140D0000"
        },
        ["Grand Inquisitor"] = {
            [0] = "00040072",
            [1] = "7B841185",
            [2] = "00000000",
            [3] = "00004656",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "130D0000"
        },
        ["Maul"] = {
            [0] = "00040072",
            [1] = "7B895525",
            [2] = "00000000",
            [3] = "00003104",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "110D0000"
        },
        ["Grand Inquisitor 2"] = {
            [0] = "00040072",
            [1] = "804B091A",
            [2] = "00000000",
            [3] = "00005909",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "130D0000"
        },
        ["Asajj Ventress"] = {
            [0] = "00040072",
            [1] = "7A1C1F46",
            [2] = "00000000",
            [3] = "00008E4D",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "080D0000"
        },
        ["Darth Sidious s2"] = {
            [0] = "00040072",
            [1] = "00000000",
            [2] = "00000000",
            [3] = "00000000",
            [4] = "0001805F",
            [5] = "18C631FF",
            [6] = "5E003000",
            [7] = "00000000",
            [8] = "00000000",
            [9] = "010D0000"
        }
    }

    -- When called without arguments, return the whole table
    if not profile_name then
        return profiles
    end

    -- Otherwise return the specific profile or wipe chip
    return profiles[profile_name] or profiles["wipe chip"]
end

function get_profile_names()
    -- Get the complete profiles table from get_profile_data
    local all_profiles = get_profile_data()

    local names = {}
    for name, _ in pairs(all_profiles) do
        table.insert(names, name)
    end
    table.sort(names)
    return names
end

function select_profile()
    local profile_names = get_profile_names()

    print("\nAvailable profiles:")
    for i, name in ipairs(profile_names) do
        print(string.format("%d. %s", i, name))
    end

    while true do
        io.write("\nSelect profile (1-" .. #profile_names .. "): ")
        local choice = tonumber(io.read())

        if choice and choice >= 1 and choice <= #profile_names then
            return profile_names[choice]
        else
            print("Invalid selection. Please try again.")
        end
    end
end

function main()
    print("\n[=== kyber crystal programmer ===]")

    -- Get profile from command line argument or prompt user
    local profile_name = args and args[1]
    if not profile_name then
        --print("\nNo profile specified as argument.")
        profile_name = select_profile()
    end

    local data_to_write = get_profile_data(profile_name)
    print("\n[+] Using profile: " .. profile_name)

    -- Display what will be written
    print("\n[+] Data to be written:")
    for addr, data in pairs(data_to_write) do
        print(string.format("Address %d: %s", addr, data))
    end

    -- Step 1: Wipe the tag
    print("\n[+] Wiping tag...")
    send_command("lf em 4x05 wipe --4305")

    -- Step 2: Write data
    print("\n[+] Writing data...")
    for addr, data in pairs(data_to_write) do
        send_command("lf em 4x05 write -a " .. addr .. " -d " .. data)
        utils.Sleep(0.5)
    end

    -- Step 3: Read back and display data for verification
    print("\n[+] Verifying writes by reading back data...")
    for addr, expected_data in pairs(data_to_write) do
        local output = send_command("lf em 4x05 read -a " .. addr)
    end

    print("\n[+] Read complete. Review output above.")
end

main()
