#include "views/settingspage.h"
#include "ui_settingspage.h"
#include <QApplication>
#include <QDesktopServices>
#include <QFileDialog>
#include <QMessageBox>
#include <QStyleHints>
#include <libnick/localization/gettext.h>
#include <libnick/system/environment.h>

using namespace Nickvision::System;
using namespace Nickvision::TubeConverter::Shared::Controllers;
using namespace Nickvision::TubeConverter::Shared::Models;

namespace Nickvision::TubeConverter::QT::Views
{
    SettingsPage::SettingsPage(const std::shared_ptr<PreferencesViewController>& controller, QWidget* parent)
        : QWidget{ parent },
        m_ui{ new Ui::SettingsPage() },
        m_controller{ controller }
    {
        m_ui->setupUi(this);
        //Localize Strings
        m_ui->tabs->setTabText(0, _("User Interface"));
        m_ui->tabs->setTabText(1, _("Downloads"));
        m_ui->tabs->setTabText(2, _("Downloader"));
        m_ui->tabs->setTabText(3, _("Converter"));
        m_ui->tabs->setTabText(4, _("aria2"));
        m_ui->lblTheme->setText(_("Theme"));
        m_ui->cmbTheme->addItem(_("Light"));
        m_ui->cmbTheme->addItem(_("Dark"));
        m_ui->cmbTheme->addItem(_("System"));
        m_ui->lblUpdates->setText(_("Automatically Check for Updates"));
        m_ui->lblCompletedNotificationTrigger->setText(_("Completed Notification Trigger"));
        m_ui->cmbCompletedNotificationTrigger->addItem(_("For each download"));
        m_ui->cmbCompletedNotificationTrigger->addItem(_("When all downloads finish"));
        m_ui->cmbCompletedNotificationTrigger->addItem(_("Never"));
        m_ui->lblPreventSuspend->setText(_("Prevent Suspend"));
        m_ui->lblRecoverCrashedDownloads->setText(_("Recover Crashed Downloads"));
        m_ui->lblDownloadImmediately->setText(_("Download Immediately After Validation"));
        m_ui->lblHistoryLength->setText(_("Download History Length"));
        m_ui->cmbHistoryLength->addItem(_("Never"));
        m_ui->cmbHistoryLength->addItem(_("One Day"));
        m_ui->cmbHistoryLength->addItem(_("One Week"));
        m_ui->cmbHistoryLength->addItem(_("One Month"));
        m_ui->cmbHistoryLength->addItem(_("Three Months"));
        m_ui->cmbHistoryLength->addItem(_("Forever"));
        m_ui->lblOverwriteExistingFiles->setText(_("Overwrite Existing Files"));
        m_ui->lblMaxNumberOfActiveDownloads->setText(_("Max Number of Active Downloads"));
        for(int i = 1; i <= 10; i++)
        {
            m_ui->cmbMaxNumberOfActiveDownloads->addItem(QString::number(i));
        }
        m_ui->lblLimitCharacters->setText(_("Limit Filename Characters"));
        m_ui->lblLimitCharacters->setToolTip(_("Restricts characters in filenames to only those supported by Windows."));
        m_ui->chkLimitCharacters->setToolTip(_("Restricts characters in filenames to only those supported by Windows."));
        m_ui->lblIncludeMediaId->setText(_("Include Media Id in Title After Validation"));
        m_ui->lblIncludeAutoGeneratedSubtitles->setText(_("Include Auto-Generated Subtitles"));
        m_ui->lblPreferredVideoCodec->setText(_("Preferred Video Codec"));
        m_ui->cmbPreferredVideoCodec->addItem(_("Any"));
        m_ui->cmbPreferredVideoCodec->addItem("VP9");
        m_ui->cmbPreferredVideoCodec->addItem("AV1");
        m_ui->cmbPreferredVideoCodec->addItem("H.264");
        m_ui->lblPreferredSubtitleFormat->setText(_("Preferred Subtitle Format"));
        m_ui->cmbPreferredSubtitleFormat->addItem(_("Any"));
        m_ui->cmbPreferredSubtitleFormat->addItem("VTT");
        m_ui->cmbPreferredSubtitleFormat->addItem("SRT");
        m_ui->cmbPreferredSubtitleFormat->addItem("ASS");
        m_ui->cmbPreferredSubtitleFormat->addItem("LRC");
        m_ui->lblVerboseLogging->setText(_("Verbose Logging"));
        m_ui->lblSponsorBlock->setText(_("Use SponsorBlock for YouTube"));
        m_ui->lblSpeedLimit->setText(_("Speed Limit"));
        m_ui->lblSpeedLimit->setToolTip(_("This limit is applied only to downloads that have enabled limiting download speed."));
        m_ui->numSpeedLimit->setToolTip(_("This limit is applied only to downloads that have enabled limiting download speed."));
        m_ui->lblProxyUrl->setText(_("Proxy URL"));
        m_ui->txtProxyUrl->setPlaceholderText(_("Enter proxy url here"));
        m_ui->lblCookiesBrowser->setText(_("Cookies from Browser"));
        m_ui->cmbCookiesBrowser->addItem(_("None"));
        m_ui->cmbCookiesBrowser->addItem(_("Brave"));
        m_ui->cmbCookiesBrowser->addItem(_("Chrome"));
        m_ui->cmbCookiesBrowser->addItem(_("Chromium"));
        m_ui->cmbCookiesBrowser->addItem(_("Edge"));
        m_ui->cmbCookiesBrowser->addItem(_("Firefox"));
        m_ui->cmbCookiesBrowser->addItem(_("Opera"));
        m_ui->cmbCookiesBrowser->addItem(_("Safari"));
        m_ui->cmbCookiesBrowser->addItem(_("Vivaldi"));
        m_ui->cmbCookiesBrowser->addItem(_("Whale"));
        m_ui->lblCookiesFile->setText(_("Cookies File"));
        m_ui->txtCookiesFile->setPlaceholderText(_("No file selected"));
        m_ui->btnSelectCookiesFile->setText(_("Select Cookies File"));
        m_ui->btnClearCookiesFile->setText(_("Clear Cookies File"));
        m_ui->lblEmbedMetadata->setText(_("Embed Metadata"));
        m_ui->lblEmbedSubtitles->setText(_("Embed Subtitles"));
        m_ui->lblEmbedChapters->setText(_("Embed Chapters"));
        m_ui->lblCropAudioThumbnails->setText(_("Crop Audio Thumbnails"));
        m_ui->lblCropAudioThumbnails->setToolTip(_("If enabled, Parabolic will crop the thumbnails of audio files to squares."));
        m_ui->chkCropAudioThumbnails->setToolTip(_("If enabled, Parabolic will crop the thumbnails of audio files to squares."));
        m_ui->lblRemoveSourceData->setText(_("Remove Source Data"));
        m_ui->lblRemoveSourceData->setToolTip(_("If enabled, Parabolic will clear metadata fields containing identifying download information."));
        m_ui->chkRemoveSourceData->setToolTip(_("If enabled, Parabolic will clear metadata fields containing identifying download information."));
        m_ui->lblPostprocessingThreads->setText(_("Postprocessing Threads"));
        for(int i = 1; i <= m_controller->getMaxPostprocessingThreads(); i++)
        {
            m_ui->cmbPostprocessingThreads->addItem(QString::number(i));
        }
        m_ui->lblUseAria->setText(_("Use aria2"));
        m_ui->lblAriaMaxConnectionsPerServer->setText(_("Max Connections Per Server (-x)"));
        m_ui->lblAriaMinSplitSize->setText(_("Minimum Split Size (-k)"));
        m_ui->lblAriaMinSplitSize->setToolTip(_("The minimum size of which to split a file (in MiB)."));
        m_ui->numAriaMinSplitSize->setToolTip(_("The minimum size of which to split a file (in MiB)."));
        //Load Settings
        DownloaderOptions options{ m_controller->getDownloaderOptions() };
        m_ui->cmbTheme->setCurrentIndex(static_cast<int>(m_controller->getTheme()));
        m_ui->chkUpdates->setChecked(m_controller->getAutomaticallyCheckForUpdates());
        m_ui->cmbCompletedNotificationTrigger->setCurrentIndex(static_cast<int>(m_controller->getCompletedNotificationPreference()));
        m_ui->chkPreventSuspend->setChecked(m_controller->getPreventSuspend());
        m_ui->chkRecoverCrashedDownloads->setChecked(m_controller->getRecoverCrashedDownloads());
        m_ui->chkDownloadImmediately->setChecked(m_controller->getDownloadImmediatelyAfterValidation());
        m_ui->cmbHistoryLength->setCurrentIndex(static_cast<int>(m_controller->getHistoryLengthIndex()));
        m_ui->chkOverwriteExistingFiles->setChecked(options.getOverwriteExistingFiles());
        m_ui->cmbMaxNumberOfActiveDownloads->setCurrentIndex(options.getMaxNumberOfActiveDownloads() - 1);
        m_ui->chkIncludeMediaId->setChecked(options.getIncludeMediaIdInTitle());
        m_ui->chkLimitCharacters->setChecked(options.getLimitCharacters());
        m_ui->chkIncludeAutoGeneratedSubtitles->setChecked(options.getIncludeAutoGeneratedSubtitles());
        m_ui->cmbPreferredVideoCodec->setCurrentIndex(static_cast<int>(options.getPreferredVideoCodec()));
        m_ui->cmbPreferredSubtitleFormat->setCurrentIndex(static_cast<int>(options.getPreferredSubtitleFormat()));
        m_ui->chkVerboseLogging->setChecked(options.getVerboseLogging());
        m_ui->chkSponsorBlock->setChecked(options.getYouTubeSponsorBlock());
        m_ui->numSpeedLimit->setValue(options.getSpeedLimit());
        m_ui->txtProxyUrl->setText(QString::fromStdString(options.getProxyUrl()));
        m_ui->cmbCookiesBrowser->setCurrentIndex(static_cast<int>(options.getCookiesBrowser()));
        m_ui->txtCookiesFile->setText(QString::fromStdString(options.getCookiesPath().filename().string()));
        m_ui->txtCookiesFile->setToolTip(QString::fromStdString(options.getCookiesPath().string()));
        m_ui->chkEmbedMetadata->setChecked(options.getEmbedMetadata());
        m_ui->chkEmbedSubtitles->setChecked(options.getEmbedSubtitles());
        m_ui->chkEmbedChapters->setChecked(options.getEmbedChapters());
        m_ui->lblCropAudioThumbnails->setEnabled(options.getEmbedMetadata());
        m_ui->chkCropAudioThumbnails->setEnabled(options.getEmbedMetadata());
        m_ui->chkCropAudioThumbnails->setChecked(options.getCropAudioThumbnails());
        m_ui->lblRemoveSourceData->setEnabled(options.getEmbedMetadata());
        m_ui->chkRemoveSourceData->setEnabled(options.getEmbedMetadata());
        m_ui->chkRemoveSourceData->setChecked(options.getRemoveSourceData());
        m_ui->cmbPostprocessingThreads->setCurrentIndex(options.getPostprocessingThreads() - 1);
        m_ui->chkUseAria->setChecked(options.getUseAria());
        m_ui->numAriaMaxConnectionsPerServer->setValue(options.getAriaMaxConnectionsPerServer());
        m_ui->numAriaMinSplitSize->setValue(options.getAriaMinSplitSize());
        if(Environment::getDeploymentMode() != DeploymentMode::Local)
        {
            m_ui->lblCookiesBrowser->setVisible(false);
            m_ui->cmbCookiesBrowser->setVisible(false);
        }
        if(Environment::getOperatingSystem() == OperatingSystem::Windows)
        {
            m_ui->lblLimitCharacters->setVisible(false);
            m_ui->chkLimitCharacters->setVisible(false);
        }
        //Signals
        connect(m_ui->cmbTheme, &QComboBox::currentIndexChanged, this, &SettingsPage::onThemeChanged);
        connect(m_ui->btnSelectCookiesFile, &QPushButton::clicked, this, &SettingsPage::selectCookiesFile);
        connect(m_ui->btnClearCookiesFile, &QPushButton::clicked, this, &SettingsPage::clearCookiesFile);
        connect(m_ui->chkEmbedMetadata, &QCheckBox::toggled, this, &SettingsPage::onEmbedMetadataChanged);
        m_ui->tabs->setCurrentIndex(0);
    }
    
    SettingsPage::~SettingsPage()
    {
        delete m_ui;
    }

    void SettingsPage::onThemeChanged(int index)
    {
        switch (static_cast<Theme>(m_ui->cmbTheme->currentIndex()))
        {
        case Theme::Light:
            QApplication::styleHints()->setColorScheme(Qt::ColorScheme::Light);
            break;
        case Theme::Dark:
            QApplication::styleHints()->setColorScheme(Qt::ColorScheme::Dark);
            break;
        default:
            QApplication::styleHints()->setColorScheme(Qt::ColorScheme::Unknown);
            break;
        }
    }

    void SettingsPage::closeEvent(QCloseEvent* event)
    {
        DownloaderOptions options{ m_controller->getDownloaderOptions() };
        m_controller->setTheme(static_cast<Theme>(m_ui->cmbTheme->currentIndex()));
        m_controller->setAutomaticallyCheckForUpdates(m_ui->chkUpdates->isChecked());
        m_controller->setCompletedNotificationPreference(static_cast<CompletedNotificationPreference>(m_ui->cmbCompletedNotificationTrigger->currentIndex()));
        m_controller->setPreventSuspend(m_ui->chkPreventSuspend->isChecked());
        m_controller->setRecoverCrashedDownloads(m_ui->chkRecoverCrashedDownloads->isChecked());
        m_controller->setDownloadImmediatelyAfterValidation(m_ui->chkDownloadImmediately->isChecked());
        m_controller->setHistoryLengthIndex(m_ui->cmbHistoryLength->currentIndex());
        options.setOverwriteExistingFiles(m_ui->chkOverwriteExistingFiles->isChecked());
        options.setMaxNumberOfActiveDownloads(m_ui->cmbMaxNumberOfActiveDownloads->currentIndex() + 1);
        options.setLimitCharacters(m_ui->chkLimitCharacters->isChecked());
        options.setIncludeMediaIdInTitle(m_ui->chkIncludeMediaId->isChecked());
        options.setIncludeAutoGeneratedSubtitles(m_ui->chkIncludeAutoGeneratedSubtitles->isChecked());
        options.setPreferredVideoCodec(static_cast<VideoCodec>(m_ui->cmbPreferredVideoCodec->currentIndex()));
        options.setPreferredSubtitleFormat(static_cast<SubtitleFormat>(m_ui->cmbPreferredSubtitleFormat->currentIndex()));
        options.setVerboseLogging(m_ui->chkVerboseLogging->isChecked());
        options.setYouTubeSponsorBlock(m_ui->chkSponsorBlock->isChecked());
        options.setSpeedLimit(m_ui->numSpeedLimit->value());
        options.setProxyUrl(m_ui->txtProxyUrl->text().toStdString());
        options.setCookiesBrowser(static_cast<Browser>(m_ui->cmbCookiesBrowser->currentIndex()));
        options.setCookiesPath(m_ui->txtCookiesFile->toolTip().toStdString());
        options.setEmbedMetadata(m_ui->chkEmbedMetadata->isChecked());
        options.setEmbedSubtitles(m_ui->chkEmbedSubtitles->isChecked());
        options.setEmbedChapters(m_ui->chkEmbedChapters->isChecked());
        options.setCropAudioThumbnails(m_ui->chkCropAudioThumbnails->isChecked());
        options.setRemoveSourceData(m_ui->chkRemoveSourceData->isChecked());
        options.setPostprocessingThreads(m_ui->cmbPostprocessingThreads->currentIndex() + 1);
        options.setUseAria(m_ui->chkUseAria->isChecked());
        options.setAriaMaxConnectionsPerServer(m_ui->numAriaMaxConnectionsPerServer->value());
        options.setAriaMinSplitSize(m_ui->numAriaMinSplitSize->value());
        m_controller->setDownloaderOptions(options);
        m_controller->saveConfiguration();
        event->accept();
    }

    void SettingsPage::selectCookiesFile()
    {
        QString file{ QFileDialog::getOpenFileName(this, _("Select Cookies File"), {}, _("TXT Files (*.txt)")) };
        if(!file.isEmpty())
        {
            std::filesystem::path path{ file.toStdString() };
            m_ui->txtCookiesFile->setText(QString::fromStdString(path.filename().string()));
            m_ui->txtCookiesFile->setToolTip(QString::fromStdString(path.string()));
        }
    }

    void SettingsPage::clearCookiesFile()
    {
        m_ui->txtCookiesFile->setText("");
        m_ui->txtCookiesFile->setToolTip("");
    }

    void SettingsPage::onEmbedMetadataChanged(bool checked)
    {
        m_ui->lblCropAudioThumbnails->setEnabled(checked);
        m_ui->chkCropAudioThumbnails->setEnabled(checked);
        m_ui->lblRemoveSourceData->setEnabled(checked);
        m_ui->chkRemoveSourceData->setEnabled(checked);
    }
}