/*
 * Copyright (C) 2005-2024 Centre National d'Etudes Spatiales (CNES)
 * Copyright (C) 2007-2012 Institut Mines Telecom / Telecom Bretagne
 *
 * This file is part of Orfeo Toolbox
 *
 *     https://www.orfeo-toolbox.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef otbKullbackLeiblerProfileImageFilter_h
#define otbKullbackLeiblerProfileImageFilter_h

#include <vector>

#include "itkArray.h"
#include "itkArray2D.h"
#include "itkVariableLengthVector.h"

#include "otbBinaryFunctorNeighborhoodVectorImageFilter.h"

namespace otb
{

/** \class CumulantsForEdgeworthProfile
  * \brief Helper class for KullbackLeiblerProfileImageFilter. Please refer to KullbackLeibleProfileImageFilter.
  *
 *
 * \ingroup OTBChangeDetection
  */
template <class TInput>
class CumulantsForEdgeworthProfile
{
public:
  typedef itk::Vector<double, 4> CumulantType;
  typedef std::vector<CumulantType> CumulantSet;
  typedef CumulantSet::iterator     Iterator;

  CumulantsForEdgeworthProfile(const TInput& input, std::vector<itk::Array2D<int>>& mask);
  virtual ~CumulantsForEdgeworthProfile()
  {
  }

  // Kullback-Leibler Profile
  template <class TInput2>
  itk::VariableLengthVector<double> KL_profile(CumulantsForEdgeworthProfile<TInput2>& cumulants);
  // Kullback-Leibler divergence at a given scale
  double KL_profile(const CumulantType& cumulants1, const CumulantType& cumulants2);
  // Access to data
  inline CumulantType& GetCumulantAtScale(int i)
  {
    return this->fCum[i];
  }
  // Check data availability
  inline bool IsDataAvailable() const
  {
    return this->fDataAvailable;
  }

  // debug
  int m_debug;

protected:
  // Momentum Estimation from encapsulated neighborhood
  int MakeSumAndMoments(const TInput& input, std::vector<itk::Array2D<int>>& mask);
  // momentum estimation from the smaller window
  int InitSumAndMoments(const TInput& input, itk::Array2D<int>& mask);
  //
  int ReInitSumAndMoments(const TInput& input, itk::Array2D<int>& mask, int level);
  // transformation moment -> cumulants (for Edgeworth)
  int MakeCumulants();

  // Internal variables
  double      fSum0, fSum1, fSum2, fSum3, fSum4;
  CumulantSet fMu;
  CumulantSet fCum;

  bool fDataAvailable;

private:
  CumulantsForEdgeworthProfile();                    // Not implemented
  CumulantsForEdgeworthProfile(const TInput& input); // Not implemented
};

namespace Functor
{
/** \class KullbackLeiblerProfile
 * \brief Functor for KullbackLeiblerProfileImageFilter. Please refer to KullbackLeiblerProfileImageFilter.
 *
 *
 * \ingroup OTBChangeDetection
 */
template <class TInput1, class TInput2, class TOutput>
class KullbackLeiblerProfile
{
public:
  KullbackLeiblerProfile();
  virtual ~KullbackLeiblerProfile()
  {
  }
  // Gives the radius min and max of neighborhood
  void SetRadius(const unsigned char& min, const unsigned char& max);
  unsigned char GetRadiusMin(void);
  unsigned char GetRadiusMax(void);
  // Gives the size of the profile
  int GetNumberOfComponentsPerPixel() const
  {
    return m_mask.size();
  }
  // functor
  TOutput operator()(const TInput1& it1, const TInput2& it2);

protected:
  // Make the set of masks to play the increase in window size
  void MakeMultiscaleProfile();
  // Internal attributes
  unsigned char                  m_RadiusMin;
  unsigned char                  m_RadiusMax;
  std::vector<itk::Array2D<int>> m_mask;
};
} // Functor

/** \class KullbackLeiblerProfileImageFilter
 * \brief Implements neighborhood-wise the computation of KullbackLeibler profile over Edgeworth approximation.
 *
 * This filter is parametrized over the types of the two
 * input images and the type of the output image.
 *
 * Numeric conversions (castings) are done by the C++ defaults.
 *
 * The filter will walk over all the pixels in the two input images, and for
 * each one of them it will do the following:
 *
 * - cast the input 1 pixel value to \c double
 * - cast the input 2 pixel value to \c double
 * - compute the first four cumulants of the two pixel values
 * - compute the value of the Edgeorth approximation of the KL distance
 * - cast the \c double value resulting to the pixel type of the output image
 * - store the casted value into the output image.
 *
 * The filter expect all images to have the same dimension (all 2D)
 *
 * See article of  Lin Saito et Levine
 * "Edgeworth Approximation of the Kullback-Leibler Distance Towards Problems in Image Analysis"
 * and
 * "Edgeworth Expansions of the Kullback-Leibler Information" (submitted to JASA, nov 25, 1999)
 * http://www.math.ucdavis.edu/~saito/publications
 *
 *  TOutput is expected to be a itk::VariableLengthVector< TPixel > and comes from an otbVectorImage< TPixel, 2 >
 *
 * \ingroup IntensityImageFilters Multithreaded
 *
 * \ingroup OTBChangeDetection
 */
template <class TInputImage1, class TInputImage2, class TOutputImage>
class ITK_EXPORT KullbackLeiblerProfileImageFilter
    : public BinaryFunctorNeighborhoodVectorImageFilter<
          TInputImage1, TInputImage2, TOutputImage,
          Functor::KullbackLeiblerProfile<typename itk::ConstNeighborhoodIterator<TInputImage1>, typename itk::ConstNeighborhoodIterator<TInputImage2>,
                                          typename TOutputImage::PixelType>>
{
public:
  /** Standard class typedefs. */
  typedef KullbackLeiblerProfileImageFilter Self;
  typedef /*typename*/ BinaryFunctorNeighborhoodVectorImageFilter<
      TInputImage1, TInputImage2, TOutputImage,
      Functor::KullbackLeiblerProfile<typename itk::ConstNeighborhoodIterator<TInputImage1>, typename itk::ConstNeighborhoodIterator<TInputImage2>,
                                      typename TOutputImage::PixelType>>
                                        Superclass;
  typedef itk::SmartPointer<Self>       Pointer;
  typedef itk::SmartPointer<const Self> ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

protected:
  KullbackLeiblerProfileImageFilter()
  {
  }
  ~KullbackLeiblerProfileImageFilter() override
  {
  }

private:
  KullbackLeiblerProfileImageFilter(const Self&) = delete;
  void operator=(const Self&) = delete;
};

} // namespace otb

#ifndef OTB_MANUAL_INSTANTIATION
#include "otbKullbackLeiblerProfileImageFilter.hxx"
#endif

#endif
