// -*- C++ -*-
//===-- remove.pass.cpp ---------------------------------------------------===//
//
// Copyright (C) Intel Corporation
//
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
// This file incorporates work covered by the following copyright and permission
// notice:
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
//
//===----------------------------------------------------------------------===//

// Test for remove, remove_if
#include "support/test_config.h"

#include _PSTL_TEST_HEADER(execution)
#include _PSTL_TEST_HEADER(algorithm)

#include "support/utils.h"

#if  !defined(_PSTL_TEST_REMOVE) && !defined(_PSTL_TEST_REMOVE_IF)
#define _PSTL_TEST_REMOVE
#define _PSTL_TEST_REMOVE_IF
#endif

using namespace TestUtils;

template <typename T>
struct run_remove
{
    template <typename Policy, typename InputIterator, typename OutputIterator, typename Size>
    void
    operator()(Policy&& exec, InputIterator first, InputIterator last, OutputIterator out_first,
               OutputIterator out_last, OutputIterator expected_first, OutputIterator expected_last, Size,
               const T& value)
    {
        // Cleaning
        ::std::copy(first, last, expected_first);
        ::std::copy(first, last, out_first);

        // Run remove
        OutputIterator i = remove(expected_first, expected_last, value);
        OutputIterator k = remove(std::forward<Policy>(exec), out_first, out_last, value);
        EXPECT_EQ(std::distance(expected_first, i), std::distance(out_first, k), "wrong return value from remove");
        EXPECT_EQ_N(expected_first, out_first, ::std::distance(expected_first, i), "wrong remove effect");
    }
};

template <typename T>
struct run_remove_if
{
    template <typename Policy, typename InputIterator, typename OutputIterator, typename Size, typename Predicate>
    void
    operator()(Policy&& exec, InputIterator first, InputIterator last, OutputIterator out_first,
               OutputIterator out_last, OutputIterator expected_first, OutputIterator expected_last, Size,
               Predicate pred)
    {
        // Cleaning
        ::std::copy(first, last, expected_first);
        ::std::copy(first, last, out_first);

        // Run remove_if
        OutputIterator i = remove_if(expected_first, expected_last, pred);
        OutputIterator k = remove_if(std::forward<Policy>(exec), out_first, out_last, pred);
        EXPECT_EQ(std::distance(expected_first, i), std::distance(out_first, k),
                  "wrong return value from remove_if");
        EXPECT_EQ_N(expected_first, out_first, ::std::distance(expected_first, i), "wrong remove_if effect");
    }
};

template <typename T, typename Predicate, typename Convert>
void
test(T trash, const T& value, Predicate pred, Convert convert)
{
    const ::std::size_t max_size = 100000;
    Sequence<T> out(max_size, [trash](size_t) { return trash; });
    Sequence<T> expected(max_size, [trash](size_t) { return trash; });

    for (size_t n = 0; n <= max_size; n = n <= 16 ? n + 1 : size_t(3.1415 * n))
    {
        Sequence<T> data(n, [&](size_t k) -> T { return convert(k); });

#ifdef _PSTL_TEST_REMOVE
        invoke_on_all_policies<>()(run_remove<T>(), data.begin(), data.end(), out.begin(), out.begin() + n,
                                   expected.begin(), expected.begin() + n, n, value);
#endif
#ifdef _PSTL_TEST_REMOVE_IF
        invoke_on_all_policies<>()(run_remove_if<T>(), data.begin(), data.end(), out.begin(), out.begin() + n,
                                   expected.begin(), expected.begin() + n, n, pred);
#endif
    }
}

struct test_non_const
{
    template <typename Policy, typename Iterator>
    void
    operator()(Policy&& exec, Iterator iter)
    {
        auto is_even = TestUtils::IsEven<float64_t>{};
        remove_if(std::forward<Policy>(exec), iter, iter, non_const(is_even));
    }
};

void test_empty_list_initialization()
{
    {
        std::vector<int> v{3,6,0,4,0,7,8,0,3,4};
        std::vector<int> expected{3,6,4,7,8,3,4};
        auto it = oneapi::dpl::remove(oneapi::dpl::execution::seq, v.begin(), v.end(), {});
        EXPECT_TRUE(it == v.begin() + 7, "not all empty list-initialized values are properly removed by oneapi::dpl::remove with `seq` policy");
        v.erase(it, v.end());
        EXPECT_TRUE(v == expected, "wrong effect from calling oneapi::dpl::remove with empty list-initialized value and with `seq` policy");
    }
    {
        std::vector<int> v{3,6,0,4,0,7,8,0,3,4};
        std::vector<int> expected{3,6,4,7,8,3,4};
        auto it = oneapi::dpl::remove(oneapi::dpl::execution::unseq, v.begin(), v.end(), {});
        EXPECT_TRUE(it == v.begin() + 7, "not all empty list-initialized values are properly removed by oneapi::dpl::remove with `unseq` policy");
        v.erase(it, v.end());
        EXPECT_TRUE(v == expected, "wrong effect from calling oneapi::dpl::remove with empty list-initialized value and with `unseq` policy");
    }

    {
        std::vector<TestUtils::DefaultInitializedToOne> v_custom{{3},{1},{5},{1},{3},{1},{8},{2},{0},{1}};
        std::vector<TestUtils::DefaultInitializedToOne> expected_custom{{3},{5},{3},{8},{2},{0}};
        auto it = oneapi::dpl::remove(oneapi::dpl::execution::par, v_custom.begin(), v_custom.end(), {});
        EXPECT_TRUE(it == v_custom.begin() + 6, "not all empty list-initialized values are properly removed by oneapi::dpl::remove with `par` policy");
        v_custom.erase(it, v_custom.end());
        EXPECT_TRUE(v_custom == expected_custom, "wrong effect from calling oneapi::dpl::remove with empty list-initialized value and with `par` policy");
    }
    {
        std::vector<TestUtils::DefaultInitializedToOne> v_custom{{3},{1},{5},{1},{3},{1},{8},{2},{0},{1}};
        std::vector<TestUtils::DefaultInitializedToOne> expected_custom{{3},{5},{3},{8},{2},{0}};
        auto it = oneapi::dpl::remove(oneapi::dpl::execution::par_unseq, v_custom.begin(), v_custom.end(), {});
        EXPECT_TRUE(it == v_custom.begin() + 6, "not all empty list-initialized values are properly removed by oneapi::dpl::remove with `par_unseq` policy");
        v_custom.erase(it, v_custom.end());
        EXPECT_TRUE(v_custom == expected_custom, "wrong effect from calling oneapi::dpl::remove with empty list-initialized value and with `par_unseq` policy");
    }
#if TEST_DPCPP_BACKEND_PRESENT
    std::vector<int> v{3,6,0,4,0,7,8,0,3,4};
    std::vector<int> expected{3,6,4,7,8,3,4};
    std::size_t idx = 0;
    {
        sycl::buffer<int> buf(v);
        auto it = oneapi::dpl::remove(oneapi::dpl::execution::dpcpp_default, oneapi::dpl::begin(buf), oneapi::dpl::end(buf), {});
        idx = it.get_idx();
        EXPECT_TRUE(idx == 7, "not all empty list-initialized values are properly removed by oneapi::dpl::remove with `device_policy` policy");
    }
    v.erase(v.begin() + idx, v.end());
    EXPECT_TRUE(v == expected, "wrong effect from calling oneapi::dpl::remove with empty list-initialized value and with `device_policy` policy");
#endif
}

int
main()
{
#if !_PSTL_ICC_18_TEST_EARLY_EXIT_MONOTONIC_RELEASE_BROKEN
    test<std::uint32_t>(666, 42, [](std::uint32_t) { return true; }, [](size_t j) { return j; });
#endif

    test<std::int32_t>(666, 2001, [](const std::int32_t& val) { return val != 2001; },
                  [](size_t j) { return ((j + 1) % 5 & 2) != 0 ? 2001 : -1 - std::int32_t(j); });
    test<float64_t>(-666.0, 8.5, [](const float64_t& val) { return val != 8.5; },
                    [](size_t j) { return ((j + 1) % 7 & 2) != 0 ? 8.5 : float64_t(j % 32 + j); });

#if !TEST_DPCPP_BACKEND_PRESENT
    test<Number>(Number(-666, OddTag()), Number(42, OddTag()), IsMultiple(3, OddTag()),
                 [](std::int32_t j) { return Number(j, OddTag()); });
#endif

#ifdef _PSTL_TEST_REMOVE_IF
    test_algo_basic_single<std::int32_t>(run_for_rnd_fw<test_non_const>());
#endif
#if !TEST_DPCPP_BACKEND_PRESENT
    test<MemoryChecker>(MemoryChecker{0}, MemoryChecker{1},
        [](const MemoryChecker& val){ return val.value() == 1; },
        [](::std::size_t idx){ return MemoryChecker{::std::int32_t(idx % 3 == 0)}; }
    );
    EXPECT_TRUE(MemoryChecker::alive_objects() == 0, "wrong effect from remove,remove_if: number of ctor and dtor calls is not equal");
#endif

    test_empty_list_initialization();

    return done();
}
