%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  This file is part of Logtalk <https://logtalk.org/>
%  SPDX-FileCopyrightText: 1998-2026 Paulo Moura <pmoura@logtalk.org>
%  SPDX-License-Identifier: Apache-2.0
%
%  Licensed under the Apache License, Version 2.0 (the "License");
%  you may not use this file except in compliance with the License.
%  You may obtain a copy of the License at
%
%      http://www.apache.org/licenses/LICENSE-2.0
%
%  Unless required by applicable law or agreed to in writing, software
%  distributed under the License is distributed on an "AS IS" BASIS,
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%  See the License for the specific language governing permissions and
%  limitations under the License.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


:- object(validation).

	:- info([
		version is 1:0:0,
		author is 'Paulo Moura',
		date is 2026-02-22,
		comment is 'Constructors for validation terms. A validation term is either ``valid(Value)`` or ``invalid(Errors)`` where ``Errors`` is a list of errors. Validation terms allow applicative-style error accumulation.',
		remarks is [
			'Type-checking support' - 'Defines a ``validation`` type for use with the ``type`` library object.'
		],
		see_also is [validation(_), validated, type, arbitrary]
	]).

	:- public(of_valid/2).
	:- mode(of_valid(@term, --nonvar), one).
	:- info(of_valid/2, [
		comment is 'Constructs a validation term holding a valid value.',
		argnames is ['Value', 'Validation']
	]).

	:- public(of_invalid/2).
	:- mode(of_invalid(@term, --nonvar), one).
	:- info(of_invalid/2, [
		comment is 'Constructs a validation term holding a single error.',
		argnames is ['Error', 'Validation']
	]).

	:- public(of_invalids/2).
	:- mode(of_invalids(@list, --nonvar), one).
	:- info(of_invalids/2, [
		comment is 'Constructs a validation term holding a list of errors.',
		argnames is ['Errors', 'Validation']
	]).

	:- public(from_goal/4).
	:- meta_predicate(from_goal(0, *, *, *)).
	:- mode(from_goal(+callable, --term, @term, --nonvar), one).
	:- info(from_goal/4, [
		comment is 'Constructs a validation term holding a value bound by calling the given goal. Otherwise returns a validation term with a single error represented by the ``Error`` argument.',
		argnames is ['Goal', 'Value', 'Error', 'Validation']
	]).

	:- public(from_goal/3).
	:- meta_predicate(from_goal(0, *, *)).
	:- mode(from_goal(+callable, --term, --nonvar), one).
	:- info(from_goal/3, [
		comment is 'Constructs a validation term holding a value bound by calling the given goal. Otherwise returns a validation term with a single error being the goal error or the atom ``fail`` representing goal failure.',
		argnames is ['Goal', 'Value', 'Validation']
	]).

	:- public(from_goal/2).
	:- meta_predicate(from_goal(1, *)).
	:- mode(from_goal(+callable, --nonvar), one).
	:- info(from_goal/2, [
		comment is 'Constructs a validation term holding a value bound by calling the given closure. Otherwise returns a validation term with a single error being the closure error or the atom ``fail`` representing closure failure.',
		argnames is ['Closure', 'Validation']
	]).

	:- public(from_generator/4).
	:- meta_predicate(from_generator(0, *, *, *)).
	:- mode(from_generator(+callable, --term, @term, --nonvar), one_or_more).
	:- info(from_generator/4, [
		comment is 'Constructs validation terms with the values generated by calling the given goal. On goal error or failure, returns a validation term with a single error represented by the ``Error`` argument.',
		argnames is ['Goal', 'Value', 'Error', 'Validation']
	]).

	:- public(from_generator/3).
	:- meta_predicate(from_generator(0, *, *)).
	:- mode(from_generator(+callable, --term, --nonvar), one_or_more).
	:- info(from_generator/3, [
		comment is 'Constructs validation terms with the values generated by calling the given goal. On goal error or failure, returns a validation term with, respectively, a single error being the goal error or the atom ``fail`` representing goal failure.',
		argnames is ['Goal', 'Value', 'Validation']
	]).

	:- public(from_generator/2).
	:- meta_predicate(from_generator(1, *)).
	:- mode(from_generator(+callable, --nonvar), one_or_more).
	:- info(from_generator/2, [
		comment is 'Constructs validation terms with the values generated by calling the given closure. On closure error or failure, returns a validation term with, respectively, a single error being the closure error or the atom ``fail`` representing closure failure.',
		argnames is ['Closure', 'Validation']
	]).

	:- public(from_optional/3).
	:- mode(from_optional(+nonvar, @term, --nonvar), one).
	:- info(from_optional/3, [
		comment is 'Converts an optional term to a validation term. Returns a valid term holding the value if the optional term is not empty. Returns an invalid term with the given error otherwise.',
		argnames is ['Optional', 'Error', 'Validation']
	]).

	:- public(from_expected/2).
	:- mode(from_expected(+nonvar, --nonvar), one).
	:- info(from_expected/2, [
		comment is 'Converts an expected term to a validation term. Returns a valid term holding the value if the expected term holds a value. Returns an invalid term with the expected term error otherwise.',
		argnames is ['Expected', 'Validation']
	]).

	:- uses(list, [
		append/3, reverse/2
	]).

	of_valid(Value, valid(Value)).

	of_invalid(Error, invalid([Error])).

	of_invalids(Errors, invalid(Errors)).

	from_goal(Goal, Value, Error, Validation) :-
		(	catch(Goal, Ball, true) ->
			(	var(Ball) ->
				Validation = valid(Value)
			;	Validation = invalid([Error])
			)
		;	Validation = invalid([Error])
		).

	from_goal(Goal, Value, Validation) :-
		(	catch(Goal, Error, true) ->
			(	var(Error) ->
				Validation = valid(Value)
			;	Validation = invalid([Error])
			)
		;	Validation = invalid([fail])
		).

	from_goal(Closure, Validation) :-
		(	catch(call(Closure, Value), Error, true) ->
			(	var(Error) ->
				Validation = valid(Value)
			;	Validation = invalid([Error])
			)
		;	Validation = invalid([fail])
		).

	from_generator(Goal, Value, Error, Validation) :-
		catch(Goal, Ball, true),
		(	var(Ball) ->
			Validation = valid(Value)
		;	Validation = invalid([Error]),
			!
		).
	from_generator(_, _, Error, invalid([Error])).

	from_generator(Goal, Value, Validation) :-
		catch(Goal, Error, true),
		(	var(Error) ->
			Validation = valid(Value)
		;	Validation = invalid([Error]),
			!
		).
	from_generator(_, _, invalid([fail])).

	from_generator(Closure, Validation) :-
		catch(call(Closure, Value), Error, true),
		(	var(Error) ->
			Validation = valid(Value)
		;	Validation = invalid([Error]),
			!
		).
	from_generator(_, invalid([fail])).

	from_optional(Optional, Error, Validation) :-
		(	Optional = optional(Value) ->
			Validation = valid(Value)
		;	Validation = invalid([Error])
		).

	from_expected(Expected, Validation) :-
		(	Expected = expected(Value) ->
			Validation = valid(Value)
		;	Expected = unexpected(Error),
			Validation = invalid([Error])
		).

	:- multifile(type::type/1).
	type::type(validation).

	:- multifile(type::check/2).
	type::check(validation, Term) :-
		(	var(Term) ->
			throw(instantiation_error)
		;	Term = valid(_) ->
			true
		;	Term = invalid(Errors) ->
			type::check(list, Errors)
		;	throw(type_error(validation, Term))
		).

:- end_object.


:- object(validation(_Validation)).

	:- info([
		version is 0:2:0,
		author is 'Paulo Moura',
		date is 2026-02-21,
		comment is 'Validation term predicates. Requires passing a validation term (constructed using the ``validation`` object predicates) as a parameter.',
		parnames is ['Validation'],
		see_also is [validation, validated]
	]).

	:- public(is_valid/0).
	:- mode(is_valid, zero_or_one).
	:- info(is_valid/0, [
		comment is 'True if the validation term holds a valid value.'
	]).

	:- public(is_invalid/0).
	:- mode(is_invalid, zero_or_one).
	:- info(is_invalid/0, [
		comment is 'True if the validation term holds one or more errors.'
	]).

	:- public(if_valid/1).
	:- meta_predicate(if_valid(1)).
	:- mode(if_valid(+callable), zero_or_more).
	:- info(if_valid/1, [
		comment is 'Applies a closure when the validation term holds a valid value using the value as argument. Succeeds otherwise.',
		argnames is ['Closure']
	]).

	:- public(if_invalid/1).
	:- meta_predicate(if_invalid(1)).
	:- mode(if_invalid(+callable), zero_or_more).
	:- info(if_invalid/1, [
		comment is 'Applies a closure when the validation term holds errors using the errors list as argument. Succeeds otherwise.',
		argnames is ['Closure']
	]).

	:- public(if_valid_or_else/2).
	:- meta_predicate(if_valid_or_else(1, 1)).
	:- mode(if_valid_or_else(+callable, +callable), zero_or_more).
	:- info(if_valid_or_else/2, [
		comment is 'Applies either ``ValidClosure`` or ``InvalidClosure`` depending on the validation term holding a value or errors.',
		argnames is ['ValidClosure', 'InvalidClosure']
	]).

	:- public(valid/1).
	:- mode(valid(--term), one_or_error).
	:- info(valid/1, [
		comment is 'Returns the value hold by the validation term. Throws an error otherwise.',
		argnames is ['Value'],
		exceptions is ['Validation term holds errors' - existence_error(valid_value,'Validation')]
	]).

	:- public(invalid/1).
	:- mode(invalid(--list), one_or_error).
	:- info(invalid/1, [
		comment is 'Returns the errors hold by the validation term. Throws an error otherwise.',
		argnames is ['Errors'],
		exceptions is ['Validation term holds a valid value' - existence_error(validation_errors,'Validation')]
	]).

	:- public(filter/3).
	:- meta_predicate(filter(1, *, *)).
	:- mode(filter(+callable, @term, --nonvar), one).
	:- info(filter/3, [
		comment is 'When the validation term holds a value and the value satisfies the closure, returns the same validation term. When the validation term holds a value that does not satisfy the closure, returns an invalid term with the given error. When the validation term holds errors, returns the same validation term.',
		argnames is ['Closure', 'Error', 'NewValidation']
	]).

	:- public(map/2).
	:- meta_predicate(map(2, *)).
	:- mode(map(+callable, --nonvar), one).
	:- info(map/2, [
		comment is 'When the validation term holds a valid value and mapping a closure with the value and the new value as additional arguments succeeds, returns a new valid term. Otherwise returns the same validation term.',
		argnames is ['Closure', 'NewValidation']
	]).

	:- public(flat_map/2).
	:- meta_predicate(flat_map(2, *)).
	:- mode(flat_map(+callable, --nonvar), one).
	:- info(flat_map/2, [
		comment is 'When the validation term holds a valid value, applies a closure with the value and the new validation term as additional arguments. Returns the new validation term on success. When the validation term holds errors, short-circuits by returning the same validation term without calling the closure. This is the monadic escape hatch for dependent steps; use ``zip/3``, ``sequence/2``, or ``traverse/3`` for error accumulation on independent steps.',
		argnames is ['Closure', 'NewValidation']
	]).

	:- public(map_or_else/3).
	:- meta_predicate(map_or_else(2, *, *)).
	:- mode(map_or_else(+callable, @term, --term), one).
	:- info(map_or_else/3, [
		comment is 'When the validation term holds a value and mapping a closure with the value and the new value as additional arguments is successful, returns the new value. Otherwise returns the given default value.',
		argnames is ['Closure', 'Default', 'Value']
	]).

	:- public(map_catching/2).
	:- meta_predicate(map_catching(2, *)).
	:- mode(map_catching(+callable, --nonvar), one).
	:- info(map_catching/2, [
		comment is 'When the validation term holds a value, applies a closure to it. Returns a valid term with the new value if the closure succeeds. Returns an invalid term with the error if the closure throws an error. Returns an invalid term with the atom ``fail`` as error if the closure fails. When the validation term holds errors, returns the same validation term.',
		argnames is ['Closure', 'NewValidation']
	]).

	:- public(map_invalid/2).
	:- meta_predicate(map_invalid(2, *)).
	:- mode(map_invalid(+callable, --nonvar), one).
	:- info(map_invalid/2, [
		comment is 'When the validation term holds errors and mapping a closure with the errors list and the new errors list as additional arguments succeeds, returns a new invalid term. Otherwise returns the same validation term.',
		argnames is ['Closure', 'NewValidation']
	]).

	:- public(map_both/3).
	:- meta_predicate(map_both(2, 2, *)).
	:- mode(map_both(+callable, +callable, --nonvar), one).
	:- info(map_both/3, [
		comment is 'When the validation term holds a value and mapping ``ValidClosure`` with the value is successful, returns a valid term with the new value. When the validation term holds errors and mapping ``InvalidClosure`` with the errors list is successful, returns an invalid term with the new errors. Otherwise returns the same validation term.',
		argnames is ['ValidClosure', 'InvalidClosure', 'NewValidation']
	]).

	:- public(swap/1).
	:- mode(swap(--nonvar), one).
	:- info(swap/1, [
		comment is 'Swaps the valid and invalid terms. If the validation term holds a value, returns an invalid term with a singleton list containing that value. If the validation term holds errors, returns a valid term with the errors list.',
		argnames is ['NewValidation']
	]).

	:- public(or/2).
	:- meta_predicate(or(*, 1)).
	:- mode(or(--term, @callable), zero_or_one).
	:- info(or/2, [
		comment is 'Returns the same validation term if it holds a value. Otherwise calls closure to generate a new validation term. Fails if the validation term holds errors and calling the closure fails or throws an error.',
		argnames is ['NewValidation', 'Closure']
	]).

	:- public(or_else/2).
	:- mode(or_else(--term, @term), one).
	:- info(or_else/2, [
		comment is 'Returns the valid value if present or the default value otherwise.',
		argnames is ['Value', 'Default']
	]).

	:- public(or_else_get/2).
	:- meta_predicate(or_else_get(*, 1)).
	:- mode(or_else_get(--term, +callable), one_or_error).
	:- info(or_else_get/2, [
		comment is 'Returns the value hold by the validation term if valid. Otherwise applies a closure to compute the value. Throws an error when the validation term holds errors and a value cannot be computed.',
		argnames is ['Value', 'Closure'],
		exceptions is ['Validation term holds errors and a value cannot be computed' - existence_error(valid_value,'Validation')]
	]).

	:- public(or_else_call/2).
	:- meta_predicate(or_else_call(*, 0)).
	:- mode(or_else_call(--term, +callable), zero_or_one).
	:- info(or_else_call/2, [
		comment is 'Returns the value hold by the validation term if valid. Calls a goal deterministically otherwise.',
		argnames is ['Value', 'Goal']
	]).

	:- public(or_else_fail/1).
	:- mode(or_else_fail(--term), zero_or_one).
	:- info(or_else_fail/1, [
		comment is 'Returns the valid value if present. Fails otherwise.',
		argnames is ['Value']
	]).

	:- public(or_else_throw/1).
	:- mode(or_else_throw(--term), one_or_error).
	:- info(or_else_throw/1, [
		comment is 'Returns the value hold by the validation term if valid. Throws the errors list hold by the validation term as an exception otherwise.',
		argnames is ['Value']
	]).

	:- public(or_else_throw/2).
	:- mode(or_else_throw(--term, @nonvar), one_or_error).
	:- info(or_else_throw/2, [
		comment is 'Returns the value hold by the validation term if valid. Throws the given error otherwise, ignoring any errors hold by the validation term.',
		argnames is ['Value', 'Error']
	]).

	:- public(zip/3).
	:- meta_predicate(zip(3, *, *)).
	:- mode(zip(+callable, +nonvar, --nonvar), one).
	:- info(zip/3, [
		comment is 'When both this validation and the other validation hold values and applying a closure with both values and the new value as additional arguments is successful, returns a valid term with the new value. When both hold errors, returns an invalid term with all errors accumulated. Otherwise returns the first invalid term.',
		argnames is ['Closure', 'OtherValidation', 'NewValidation']
	]).

	:- public(flatten/1).
	:- mode(flatten(--nonvar), one).
	:- info(flatten/1, [
		comment is 'Flattens a nested validation term. When the validation term holds a value that is itself a validation term, returns the inner validation term. When the validation term holds a non-validation value, returns the same validation term. When the validation term holds errors, returns the same validation term.',
		argnames is ['NewValidation']
	]).

	:- public(to_optional/1).
	:- mode(to_optional(--nonvar), one).
	:- info(to_optional/1, [
		comment is 'Converts the validation term to an optional term. Returns a non-empty optional term holding the value if the validation term holds a value. Returns an empty optional term if the validation term holds errors.',
		argnames is ['Optional']
	]).

	:- public(to_expected/1).
	:- mode(to_expected(--nonvar), one).
	:- info(to_expected/1, [
		comment is 'Converts the validation term to an expected term. Returns an expected term holding the value if the validation term holds a value. Returns an expected term with the errors list as the unexpected error otherwise.',
		argnames is ['Expected']
	]).

	:- uses(list, [append/3]).

	is_valid :-
		parameter(1, valid(_)).

	is_invalid :-
		parameter(1, invalid(_)).

	if_valid(Closure) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			call(Closure, Value)
		;	true
		).

	if_invalid(Closure) :-
		parameter(1, Validation),
		(	Validation = invalid(Errors) ->
			call(Closure, Errors)
		;	true
		).

	if_valid_or_else(ValidClosure, InvalidClosure) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			call(ValidClosure, Value)
		;	Validation = invalid(Errors),
			call(InvalidClosure, Errors)
		).

	valid(Value) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	existence_error(valid_value, Validation)
		).

	invalid(Errors) :-
		parameter(1, Validation),
		(	Validation = invalid(Errors) ->
			true
		;	existence_error(validation_errors, Validation)
		).

	filter(Closure, Error, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			(	call(Closure, Value) ->
				NewValidation = Validation
			;	NewValidation = invalid([Error])
			)
		;	NewValidation = Validation
		).

	map(Closure, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value),
			catch(call(Closure, Value, NewValue), _, fail) ->
			NewValidation = valid(NewValue)
		;	NewValidation = Validation
		).

	flat_map(Closure, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value),
			catch(call(Closure, Value, NewValidation), _, fail) ->
			true
		;	NewValidation = Validation
		).

	map_or_else(Closure, Default, Value) :-
		parameter(1, Validation),
		(	Validation = valid(Term),
			catch(call(Closure, Term, Value), _, fail) ->
			true
		;	Value = Default
		).

	map_catching(Closure, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			(	catch(call(Closure, Value, NewValue), Error, true) ->
				(	var(Error) ->
					NewValidation = valid(NewValue)
				;	NewValidation = invalid([Error])
				)
			;	NewValidation = invalid([fail])
			)
		;	NewValidation = Validation
		).

	map_invalid(Closure, NewValidation) :-
		parameter(1, Validation),
		(	Validation = invalid(Errors) ->
			(	catch(call(Closure, Errors, NewErrors), _, fail) ->
				NewValidation = invalid(NewErrors)
			;	NewValidation = Validation
			)
		;	NewValidation = Validation
		).

	map_both(ValidClosure, InvalidClosure, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			(	catch(call(ValidClosure, Value, NewValue), _, fail) ->
				NewValidation = valid(NewValue)
			;	NewValidation = Validation
			)
		;	Validation = invalid(Errors),
			(	catch(call(InvalidClosure, Errors, NewErrors), _, fail) ->
				NewValidation = invalid(NewErrors)
			;	NewValidation = Validation
			)
		).

	swap(NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			NewValidation = invalid([Value])
		;	Validation = invalid(Errors),
			NewValidation = valid(Errors)
		).

	or(NewValidation, Closure) :-
		parameter(1, Validation),
		(	Validation = valid(_) ->
			NewValidation = Validation
		;	catch(call(Closure, NewValidation), _, fail)
		).

	or_else(Value, Default) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	Value = Default
		).

	or_else_get(Value, Closure) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	catch(call(Closure, Value), _, existence_error(valid_value, Validation)) ->
			true
		;	existence_error(valid_value, Validation)
		).

	or_else_call(Value, Goal) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	once(Goal)
		).

	or_else_fail(Value) :-
		parameter(1, valid(Value)).

	or_else_throw(Value) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	Validation = invalid(Errors),
			throw(Errors)
		).

	or_else_throw(Value, Error) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			true
		;	throw(Error)
		).

	zip(Closure, OtherValidation, NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Value1),
			OtherValidation = valid(Value2),
			catch(call(Closure, Value1, Value2, NewValue), _, fail) ->
			NewValidation = valid(NewValue)
		;	Validation = invalid(Errors1),
			OtherValidation = invalid(Errors2) ->
			append(Errors1, Errors2, AllErrors),
			NewValidation = invalid(AllErrors)
		;	Validation = invalid(_) ->
			NewValidation = Validation
		;	NewValidation = OtherValidation
		).

	flatten(NewValidation) :-
		parameter(1, Validation),
		(	Validation = valid(Inner),
			(Inner = valid(_) ; Inner = invalid(_)) ->
			NewValidation = Inner
		;	NewValidation = Validation
		).

	to_optional(Optional) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			Optional = optional(Value)
		;	Optional = empty
		).

	to_expected(Expected) :-
		parameter(1, Validation),
		(	Validation = valid(Value) ->
			Expected = expected(Value)
		;	Validation = invalid(Errors),
			Expected = unexpected(Errors)
		).

:- end_object.
