# Copyright (C) 2021 Riccardo Sturani
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http: //www.gnu.org/licenses/>.

"""Regression tests for the SimInspiralSpinTaylorOrbitalDriver() function of
lalsimulation/src/LALSimInspiralSpinTaylor.c.
"""

import sys

from pathlib import Path

import numpy as np

import pytest

import lal
import lalsimulation as lalsim

DEFAULT_FILE = Path(__file__).parent / 'reviewed_evolveorbit.ini'
NEW_DATA_STR = '######### NEW DATASET #############'


def _format_dataset(dataset):
    parameters = dataset['parameters']
    spindata = dataset['spin-data']
    dataset = (
        parameters,
        (spindata["spin1x"], spindata["spin1y"], spindata["spin1z"]),
        (spindata["spin2x"], spindata["spin2y"], spindata["spin2z"]),
    )
    try:
        return pytest.param(*dataset, id=parameters['approximant'])
    except AttributeError:  # pytest < 3.1.0
        return dataset

def iter_datasets_from_config(filename):
    with open(str(filename), "r") as file:
        dataset = {}
        for line in map(str.strip, file):
            if not line:
                continue
            if line == NEW_DATA_STR:
                if dataset:  # emit the dataset
                    yield _format_dataset(dataset)
                dataset = {}
            # parse new section header
            elif line.startswith("["):
                section = line.strip("[]")
                dataset[section] = {}
            # parse data
            else:
                (key, val) = map(str.strip, line.split('=', 1))
                if section == "spin-data":
                    val = np.fromstring(val, dtype=float, sep=" ")
                dataset[section][key] = val
        # emit the last dataset
        yield _format_dataset(dataset)


@pytest.mark.parametrize(
    ("pars", "S1", "S2"),
    iter_datasets_from_config(DEFAULT_FILE),
)
def test_spintaylor_pnevolveorbit(pars, S1, S2):
    """Regression test for lalsim.SimInspiralSpinTaylorPNEvolveOrbit.
    """
    # Fixed parameters read from reviewed_evolveorbit.ini generated by GenerateOrbitalReference.py
    lal_pars = lal.CreateDict()
    lalsim.SimInspiralWaveformParamsInsertPNPhaseOrder(lal_pars, int(pars["phaseO"]))
    lalsim.SimInspiralWaveformParamsInsertPNSpinOrder(lal_pars, int(pars["spinO"]))
    lalsim.SimInspiralWaveformParamsInsertPNTidalOrder(lal_pars, int(pars["tidalO"]))

    approx = lalsim.SimInspiralGetApproximantFromString(pars['approximant'])
    out = lalsim.SimInspiralSpinTaylorOrbitalDriver(
        float(pars['phiref']),
        float(pars['deltaT']),
        float(pars['m1']) * lal.MSUN_SI,
        float(pars['m2']) * lal.MSUN_SI,
        float(pars['fstart']),
        float(pars['fref']),
        float(pars['s1x']),
        float(pars['s1y']),
        float(pars['s1z']),
        float(pars['s2x']),
        float(pars['s2y']),
        float(pars['s2z']),
        float(pars['lnhatx']),
        float(pars['lnhaty']),
        float(pars['lnhatz']),
        float(pars['e1x']),
        float(pars['e1y']),
        float(pars['e1z']),
        lal_pars,
        approx,
    )
    spin1out = out[2:5]
    spin2out = out[5:8]

    for spinin, spinout in (
        (S1, spin1out),
        (S2, spin2out),
    ):
        for i in range(3):
            np.testing.assert_allclose(
                spinout[i].data.data,
                spinin[i],
                err_msg="{} test failed.".format(approx),
                rtol=1e-7,
            )


if __name__ == '__main__':
    args = sys.argv[1:] or ["-v", "-rs", "--junit-xml=junit-spintaylor-evolveorbit.xml"]
    sys.exit(pytest.main(args=[__file__] + args))
