# Copyright (C) 2021 N. K. Johnson-McDaniel
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http: //www.gnu.org/licenses/>.

"""
Regression tests for the precession-averaged evolution in lalsimulation.tilts_at_infinity
Data generated by <https://git.ligo.org/waveforms/reviews/spin-tilt-angles-at-infinity/-/blob/master/generate_prec_avg_evol_regression_data.py>
Based on test_nrfits.py
"""

import sys
import pytest
import numpy as np

from lalsimulation.tilts_at_infinity import prec_avg_tilt_comp

# -- regression data for tilts at infinity ---------------------

# Format: (m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, tilt1_inf, tilt2_inf, version)
# There are 10 completely random cases, 3 stable aligned-spin cases, and 3 single-spin/nonspinning cases
test_data_inf = [
    (7.2806372691932391e+31, 4.5637789475827883e+31, 0.0780559692328455, 0.6739600779634284, 2.6937108685379725, 2.6025031930498699, 0.2697097825001201, 25.7799586687455502, 2.5323883843612842, 2.6357457204361951, 'v1'),
    (1.5903682138906522e+32, 6.6605591113503082e+31, 0.3507421278339221, 0.7472239781363749, 0.2277134593464859, 2.5920427275438933, 3.0765514007793553, 22.4900378678221600, 0.3479377633288823, 2.5233814066721330, 'v1'),
    (1.0406555317689320e+32, 2.2576530183169390e+31, 0.4672064855320761, 0.7400042329641560, 1.7469211240711668, 2.4933320819908849, 3.5389623014824338, 5.5383108125765368, 1.7689674276391143, 2.4005245969349387, 'v1'),
    (1.6668852843911248e+32, 7.2460401374588622e+31, 0.8904143842706523, 0.2327177391526354, 0.5808359488179210, 1.9543003765264826, 4.5003889243933646, 23.0967145837193044, 0.6176132278576946, 1.8191364873020248, 'v1'),
    (1.6313340698382602e+32, 1.5155960799808479e+32, 0.1598971410526803, 0.6984127562036798, 0.0102379656588668, 2.6178562513580887, 5.1927858027133080, 9.8714194280043497, 1.9691695383266792, 2.1240681245588560, 'v1'),
    (1.4173302171970056e+32, 1.8963007373188665e+32, 0.3590367528945194, 0.0212923290518682, 1.0395435049796435, 1.6071179842844330, 2.4995349322883498, 13.0512623199863125, 1.0270238516853196, 1.7741858180707031, 'v1'),
    (1.5706410067106716e+32, 1.1813080330602199e+32, 0.9277147448545591, 0.9831769827655591, 1.6194139035912902, 2.2919716674989243, 4.8762649271489922, 26.7118421561182977, 1.6829997582165392, 2.2385102880266547, 'v1'),
    (1.7053346102920745e+32, 1.8499621608732496e+32, 0.7868899230796924, 0.0745511706270313, 0.7685241859277815, 0.5221195279074894, 1.8330632928388244, 9.3002097809822075, 0.7460474899052507, 0.8196652568643666, 'v1'),
    (6.2849097914864694e+31, 9.2436581531221736e+31, 0.2697591025697903, 0.9757159979545857, 0.3303596447221471, 1.6651377913023990, 0.0124474769515431, 29.9623100554478334, 0.9899622000905941, 1.6079258874117732, 'v1'),
    (9.1832452499310014e+31, 1.4214106390734381e+32, 0.1499746249450188, 0.3794444707996283, 3.1371997316199884, 2.4560896871453868, 3.5363360936306676, 5.6724285537795707, 3.0162004808726675, 2.4605098291667908, 'v1'),
    (1.1083382555836517e+32, 4.0369149567869318e+31, 0.4087362583921147, 0.1086981510534554, 0.0000000000000000, 0.0000000000000000, 4.0683789316198142, 50.8308320694211559, 0.0000000000000000, 0.0000000000000000, 'v1'),
    (1.4178127560341587e+32, 5.1586567260882585e+31, 0.3980294568299950, 0.4823750965056239, 3.1415926535897931, 0.0000000000000000, 1.4813897501195596, 17.5633445331161084, 3.1415926535897931, 0.0000000000000000, 'v1'),
    (4.0343196540527723e+31, 1.7195362633237466e+32, 0.1084069818436840, 0.2289123794537044, 3.1415926535897931, 3.1415926535897931, 0.8605133529125735, 5.9271498814314931, 3.1415926535897931, 3.1415926535897931, 'v1'),
    (1.2540284829006703e+32, 3.7182028194541439e+31, 0.0000000000000000, 0.5667356118922166, np.nan, 2.2399350724868898, np.nan, 11.2747888609023583, np.nan, 2.2399350724868898, 'v1'),
    (1.3531891236505135e+32, 1.0433384943250556e+32, 0.9482347783189793, 0.0000000000000000, 2.0162052392931540, np.nan, np.nan, 18.6770747003028958, 2.0162052392931540, np.nan, 'v1'),
    (1.4094859632880357e+32, 7.4592772668463729e+31, 0.0000000000000000, 0.0000000000000000, np.nan, np.nan, np.nan, 29.6944086574583750, np.nan, np.nan, 'v1')
]

# Three completely random cases with the mpmath evolution
test_data_inf_mpmath = [
    (1.2744453996367859e+32, 5.2544363449444431e+31, 0.3118883084796787, 0.6971558458503893, 1.8116828717669373, 2.7841391708757288, 5.6565071738150836, 44.5045151787982931, 1.7924875705097232, 2.9061164711559715, 'v1'),
    (1.1613311277436176e+32, 7.3041667088762486e+31, 0.1115397410219834, 0.7363796544786456, 0.3873124965997035, 2.3078085468489320, 1.0901268131552564, 14.2939588713552066, 0.3621658619605687, 2.3145414912464926, 'v1'),
    (1.5584723246438802e+32, 1.9739944542478597e+32, 0.3729598440714913, 0.7984647762961395, 2.6031916801411876, 2.1728661643930294, 3.2658765571056700, 9.9962485817466806, 2.1476552636307251, 2.3261431995543225, 'v1')
]

# -- regression data for tilts at finite L ---------------------

# Format: (m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf, tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg, version)
# There are 10 completely random cases, 3 stable aligned-spin cases, and 3 single-spin/nonspinning cases
test_data_Lf = [
    (1.3693634429388840e+32, 5.1945227214983403e+31, 0.2811395119455867, 0.7825889405195621, 1.4404895401963882, 2.3217281199176965, 4.6226578042297399, 13.5597501829451303, 263288.4421777905081399, 1.4569476506689862, 1.4569481876674983e+00, 1.4569479191682464, 2.3066132207491719, 2.3066139023497665, 2.3066135615494168, 'v1'),
    (1.2853989801815505e+32, 1.0766211268759188e+32, 0.1010896017003354, 0.5559633886911199, 2.2036611657875875, 1.6929986039340694, 2.3303866013196499, 17.6954342790123036, 469343872.8167735934257507, 2.4970150723723443, 2.4970150753926719e+00, 2.4970150738825079, 1.6522727308064866, 1.6522727312017680, 1.6522727310041274, 'v1'),
    (1.7389625338281511e+32, 1.6227354098502751e+31, 0.8841436736367757, 0.6238202780723805, 0.5606474341352679, 0.3953341243883686, 0.3755243096174243, 26.6361379568663565, 1594489521705.8554687500000000, 0.5644563527412929, 5.6445635274129669e-01, 0.5644563527412946, 0.6038157650046003, 0.6038157650046525, 0.6038157650046285, 'v1'),
    (9.3935573512926824e+31, 7.4406948841141732e+31, 0.9434146611758947, 0.9397564757508704, 0.0837846638466810, 2.0498519141172662, 4.1309630074989760, 50.6481806137220829, 117917111.0059724897146225, 1.0904552114290622, 1.0904552260951237e+00, 1.0904552187620931, 1.3662120768815273, 1.3662120937165663, 1.3662120852990467, 'v1'),
    (8.5456279267317436e+31, 3.0633552719515871e+31, 0.3375620052234589, 0.3079101375483646, 0.8665873932099059, 0.2769690323546233, 5.7230378984871706, 41.8312273717860279, 32199528295126.9062500000000000, 0.8577554095721789, 8.5775540957217977e-01, 0.8577554095721793, 0.3884081286316906, 0.3884081286316962, 0.3884081286316932, 'v1'),
    (9.5542775954690621e+31, 8.2034331736707562e+31, 0.5325245476593115, 0.4148966679687992, 1.0938274443689766, 0.3906800527131178, 0.8869879530642922, 27.6408051105592705, 728710.7676898157224059, 0.8221959621787063, 8.2219736303210578e-01, 0.8221966626056340, 0.9527616301222653, 0.9527635125524638, 0.9527625713376794, 'v1'),
    (1.5573775226602199e+32, 7.3366824761580696e+31, 0.5678848884177591, 0.7243471187962698, 2.8627400297859804, 2.8369273158279182, 2.9667543069034812, 22.5199814325359924, 11.6151582078317386, 2.9072904981379120, 2.9158737103320909e+00, 2.9115427376023906, 2.7709578212760819, 2.7800554995990661, 2.7754797183322513, 'v1'),
    (4.5280495732393712e+31, 1.3578511572661605e+32, 0.5006964545351558, 0.2363147213856648, 0.1438612129441551, 1.8215416847345360, 5.8027474877892660, 41.4358150405961609, 172158.9196273836423643, 0.2392727490336007, 2.3927349818344315e-01, 0.2392731236088095, 1.8169759505261540, 1.8169760798708456, 1.8169760151984993, 'v1'),
    (1.3373153149775163e+32, 2.3615694120472288e+31, 0.0406582752063830, 0.8809151675738949, 2.8511187119954635, 0.4802766110289497, 1.8039419192276962, 46.8096895851636248, 23335.8571301748816040, 2.8558370551639953, 2.8558380117932027e+00, 2.8558375334782098, 0.4815639091333803, 0.4815640613010738, 0.4815639852172327, 'v1'),
    (7.8044861325535998e+31, 1.5945028461046732e+32, 0.5569190122247453, 0.7867873984724707, 3.1329103876287685, 1.3041739065692981, 1.4297532876557435, 28.4667843946457140, 8777.8088143169607065, 2.6081532118774109, 2.6082291683874455e+00, 2.6081911889111251, 1.3727040866932574, 1.3727177343035355, 1.3727109105030697, 'v1'),
    (4.1772379456277682e+31, 6.0272680729023315e+31, 0.5355738691407114, 0.0733929622800233, 0.0000000000000000, 0.0000000000000000, 1.4445333655673247, 24.8462043614046166, 73266280119.8235015869140625, 0.0000000000000000, 0.0000000000000000e+00, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 'v1'),
    (7.5669704572683208e+31, 3.5828538489095214e+31, 0.9031589774827424, 0.6574707388837899, 3.1415926535897931, 0.0000000000000000, 3.9355045878004122, 64.0174520298004097, 429.6281855058211363, 3.1415926535897931, 3.1415926535897931e+00, 3.1415926535897931, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 'v1'),
    (1.2248542544499242e+32, 1.2500604444998141e+32, 0.8618743548641040, 0.5852173640574340, 3.1415926535897931, 3.1415926535897931, 0.8498950520396730, 9.9099873326398793, 63168107847.6711120605468750, 3.1415926535897931, 3.1415926535897931e+00, 3.1415926535897931, 3.1415926535897931, 3.1415926535897931, 3.1415926535897931, 'v1'),
    (9.5043903378966965e+31, 1.5624374576050841e+31, 0.0000000000000000, 0.2188314494012674, np.nan, 1.3077967832896478, np.nan, 39.6159033055111678, 145.6988133632754909, np.nan, np.nan, np.nan, 1.3077967832896478, 1.3077967832896478, 1.3077967832896478, 'v1'),
    (1.9173842811693528e+32, 1.9356335171636841e+32, 0.3864629450200214, 0.0000000000000000, 1.0047044868348598, np.nan, np.nan, 9.7612298683346044, 5872596272746.4580078125000000, 1.0047044868348598, 1.0047044868348598e+00, 1.0047044868348598, np.nan, np.nan, np.nan, 'v1'),
    (1.2464896389287560e+31, 1.9069948698909246e+32, 0.0000000000000000, 0.0000000000000000, np.nan, np.nan, np.nan, 33.8470570058343370, 3457086980.6832594871520996, np.nan, np.nan, np.nan, np.nan, np.nan, np.nan, 'v1')
]

# Three completely random cases with the mpmath evolution
test_data_Lf_mpmath = [
    (2.1749094501394304e+31, 4.2803971461615173e+31, 0.8171443399797673, 0.0960259315076889, 2.7963042313803617, 2.7969847960258400, 4.9187387118857870, 106.1459103386688838, 31.6492272092781377, 2.8056131329383169, 2.8066142058851327e+00, 2.8061133103727616, 2.7695734049187006, 2.7735136165504746, 2.7715385070604133, 'v1'),
    (6.6024375918486194e+31, 1.3581653318327119e+32, 0.2896482093737192, 0.1557731560879527, 0.0032736973686186, 0.4697152127409246, 6.1005880467106461, 35.0688117281698055, 17.0854023419982433, 0.0448252839247875, 4.8358944092488206e-02, 0.0466255781848872, 0.4693426919493695, 0.4696714998395489, 0.4695071225078533, 'v1'),
    (6.8273518725273872e+31, 1.9785127782041672e+32, 0.9264424055066628, 0.1627701412711647, 1.0434021046498967, 0.0943400387120294, 6.0385933467159330, 24.5191350051303871, 104.1445252089477123, 1.0533778494929313, 1.0534105103797378e+00, 1.0533941800122715, 0.0351085115153262, 0.0366624466242567, 0.0358938856537473, 'v1')
]

# -- test functions ---------------------

@pytest.mark.parametrize("m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, tilt1_inf, tilt2_inf, version", test_data_inf)
def test_inf(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, tilt1_inf, tilt2_inf, version):
    """
    Regression test for prec_avg_tilt_comp for calculating tilts at infinity

    m1, m2: Detector frame masses of the binary, in kg
    chi1, chi2: Dimensionless spin magnitudes of the binary
    tilt1, tilt2: Tilt angles of the binary's spins (w.r.t. the orbital angular momentum) at fref
    phi12: Angle between the in-plane components of the spins at fref
    fref: Reference frequency, in Hz
    tilt1_inf, tilt2_inf: Tilt angles at infinity
    version: Version of calculation to test
    """

    res = prec_avg_tilt_comp(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, version=version)

    rtol = 1.e-6
    np.testing.assert_allclose([res['tilt1_inf'], res['tilt2_inf']], [tilt1_inf, tilt2_inf], \
        err_msg="Check of tilts at infinity failed.", rtol=rtol)

@pytest.mark.parametrize("m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, tilt1_inf, tilt2_inf, version", test_data_inf_mpmath)
def test_inf_mpmath(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, tilt1_inf, tilt2_inf, version):
    """
    Regression test for prec_avg_tilt_comp for calculating tilts at infinity with the mpmath evolution

    m1, m2: Detector frame masses of the binary, in kg
    chi1, chi2: Dimensionless spin magnitudes of the binary
    tilt1, tilt2: Tilt angles of the binary's spins (w.r.t. the orbital angular momentum) at fref
    phi12: Angle between the in-plane components of the spins at fref
    fref: Reference frequency, in Hz
    tilt1_inf, tilt2_inf: Tilt angles at infinity
    version: Version of calculation to test
    """

    # Skip this test if mpmath is not available
    pytest.importorskip("mpmath")

    res = prec_avg_tilt_comp(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, method="odefun", version=version)

    rtol = 1.e-14
    np.testing.assert_allclose([res['tilt1_inf'], res['tilt2_inf']], [tilt1_inf, tilt2_inf], \
        err_msg="Check of tilts at infinity with mpmath evolution failed.", rtol=rtol)

@pytest.mark.parametrize("m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf, tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg, version", test_data_Lf)
def test_Lf(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf, tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg, version):
    """
    Regression test for prec_avg_tilt_comp for calculating tilts at finite separation

    m1, m2: Detector frame masses of the binary, in kg
    chi1, chi2: Dimensionless spin magnitudes of the binary
    tilt1, tilt2: Tilt angles of the binary's spins (w.r.t. the orbital angular momentum) at fref
    phi12: Angle between the in-plane components of the spins at fref
    fref: Reference frequency, in Hz
    Lf: Final magnitude of orbital angular momentum in total mass = 1 units for output at finite separation
    tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg: Minimum, maximum, and average tilts at separation
    version: Version of calculation to test
    """

    res = prec_avg_tilt_comp(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf=Lf, version=version)

    rtol = 1.e-6
    np.testing.assert_allclose([res['tilt1_sep_min'], res['tilt2_sep_min']], [tilt1_sep_min, tilt2_sep_min], \
        err_msg="Check of minimum tilts at finite separation failed.", rtol=rtol)
    np.testing.assert_allclose([res['tilt1_sep_max'], res['tilt2_sep_max']], [tilt1_sep_max, tilt2_sep_max], \
        err_msg="Check of maximum tilts at finite separation failed.", rtol=rtol)
    np.testing.assert_allclose([res['tilt1_sep_avg'], res['tilt2_sep_avg']], [tilt1_sep_avg, tilt2_sep_avg], \
        err_msg="Check of average tilts at finite separation failed.", rtol=rtol)

@pytest.mark.parametrize("m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf, tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg, version", test_data_Lf_mpmath)
def test_Lf_mpmath(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf, tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg, version):
    """
    Regression test for prec_avg_tilt_comp for calculating tilts at finite separation

    m1, m2: Detector frame masses of the binary, in kg
    chi1, chi2: Dimensionless spin magnitudes of the binary
    tilt1, tilt2: Tilt angles of the binary's spins (w.r.t. the orbital angular momentum) at fref
    phi12: Angle between the in-plane components of the spins at fref
    fref: Reference frequency, in Hz
    Lf: Final magnitude of orbital angular momentum in total mass = 1 units for output at finite separation
    tilt1_sep_min, tilt1_sep_max, tilt1_sep_avg, tilt2_sep_min, tilt2_sep_max, tilt2_sep_avg: Minimum, maximum, and average tilts at separation
    version: Version of calculation to test
    """

    # Skip this test if mpmath is not available
    pytest.importorskip("mpmath")

    res = prec_avg_tilt_comp(m1, m2, chi1, chi2, tilt1, tilt2, phi12, f0, Lf=Lf, method="odefun", version=version)

    rtol = 1.e-12
    np.testing.assert_allclose([res['tilt1_sep_min'], res['tilt2_sep_min']], [tilt1_sep_min, tilt2_sep_min], \
        err_msg="Check of minimum tilts at finite separation with mpmath evolution failed.", rtol=rtol)
    np.testing.assert_allclose([res['tilt1_sep_max'], res['tilt2_sep_max']], [tilt1_sep_max, tilt2_sep_max], \
        err_msg="Check of maximum tilts at finite separation with mpmath evolution failed.", rtol=rtol)
    np.testing.assert_allclose([res['tilt1_sep_avg'], res['tilt2_sep_avg']], [tilt1_sep_avg, tilt2_sep_avg], \
        err_msg="Check of average tilts at finite separation with mpmath evolution failed.", rtol=rtol)

# -- run the tests ------------------------------
if __name__ == '__main__':
    args = sys.argv[1:] or ["-v", "-rs", "--junit-xml=junit-prec_avg_evol.xml"]
    sys.exit(pytest.main(args=[__file__] + args))
