
#include "gtk-local.h"
#include "gdk/gdkkeysyms.h"

#include "zoom-tool.h"
#include "gtk-graph.h"
#include "gtk-toolbar.h"

#ifdef GAG_USE_GRAB
static int _use_grab_pointer = 1;
static int _use_grab_keyboard = 1;
#else
static int _use_grab_pointer = 0;
static int _use_grab_keyboard = 0;
#endif

typedef enum _zoom_drawing_mode {
    _ZOOM_FIRST, 
    _ZOOM_INTERMEDIATE, 
    _ZOOM_LAST, 
    _ZOOM_DRAW
} _zoom_drawing_mode_t;

static void _draw_zoom_cursor( gtv_zoom_data_t *data, gint x, gint y,
 _zoom_drawing_mode_t mode)
{
    if (mode != _ZOOM_DRAW) {
        *data->xpos = x;
        *data->ypos = y;
    }
    if (mode == _ZOOM_INTERMEDIATE) { // "undo" draw on last position
        _draw_zoom_cursor( data, *data->xpos, *data->ypos, _ZOOM_DRAW);
        return;
    } 
    set_cursor((ggtk_env_t*)data->env, x, y, TRUE, data->width, data->height);
}

static void _draw_zoom_cursor_handler( gpointer user_data)
{
    gtv_zoom_data_t *data = (gtv_zoom_data_t *)user_data;
    _draw_zoom_cursor( data, *data->xpos, *data->ypos, _ZOOM_DRAW);
}

static void ggtk_ungrab( ggtk_env_t *genv)
{
    if (_use_grab_keyboard || _use_grab_pointer) {
        GdkDisplay *display = gtk_widget_get_display(genv->drawing_area);
        GdkSeat   *seat    = gdk_display_get_default_seat(display);
        gdk_seat_ungrab(seat);
    }
}

void ggtk_deactivate_zoom( gtv_zoom_data_t *data)
{
    ggtk_env_t *genv = (ggtk_env_t *)data->env;
    if (data->command) {
        _draw_zoom_cursor( data, *data->xpos, *data->ypos, _ZOOM_LAST);
    }
    set_cursor((ggtk_env_t*)data->env, 0, 0, FALSE, 0, 0);
    ggtk_set_pointer_event_handler( NULL, NULL);
    ggtk_set_post_refresh_handler(genv, NULL, NULL);
    ggtk_ungrab( genv);
    sic_post_widget_created();
}

static int _zoom_handler( GdkEvent *event, gpointer event_data, gpointer user_data)
{
    gtv_zoom_data_t *zoom_data = (gtv_zoom_data_t *)user_data;
    ggtk_env_t *genv = (ggtk_env_t *)zoom_data->env;
    GdkWindow *window = gtk_widget_get_window(genv->drawing_area);

    if (event->type == GDK_BUTTON_PRESS) {
        GdkEventButton *e = (GdkEventButton *)event;
        int width = gtk_widget_get_allocated_width(genv->drawing_area);
        int height = gtk_widget_get_allocated_height(genv->drawing_area);
        gboolean inside_drawing_area = (e->x >= 0.0 && e->y >= 0.0 &&
            e->x < (gdouble)width && e->y < (gdouble)height);

        /*
         * When the pointer is grabbed (zoom mode), some window-manager actions
         * (e.g. clicking the titlebar) can still be delivered as button events
         * on the drawing area, but with coordinates outside its bounds.
         * Ignore those clicks so they don't terminate the crosshair/zoom mode.
         */
        if (e->window == window && inside_drawing_area) {
            switch (e->button) {
                case 1:  *zoom_data->command = '^'; break;
                case 2:  *zoom_data->command = '&'; break;
                case 3:  *zoom_data->command = '*'; break;
            }
            ggtk_deactivate_zoom(zoom_data);
        } else {
            ggtk_ungrab( genv);
            gtk_widget_add_events( genv->drawing_area, GDK_POINTER_MOTION_MASK);
        }
        return TRUE; 
    } else if (event->type == GDK_MOTION_NOTIFY) {
        GdkEventMotion *e = (GdkEventMotion *)event;
        _draw_zoom_cursor(zoom_data, e->x, e->y, _ZOOM_INTERMEDIATE);
        return TRUE;
    } else if (event->type == GDK_KEY_PRESS) {
        GdkEventKey *e = (GdkEventKey *)event;
        int xo = 0;
        int yo = 0;

        switch (e->keyval) {
        case GDK_KEY_Left:
            xo = -1;
            break;
        case GDK_KEY_Right:
            xo = 1;
            break;
        case GDK_KEY_Up:
            yo = -1;
            break;
        case GDK_KEY_Down:
            yo = 1;
            break;
        }
        if (xo || yo) {
            GdkDisplay *display = gtk_widget_get_display(genv->drawing_area);
            GdkSeat *seat = gdk_display_get_default_seat(display);
            GdkDevice *pointer = gdk_seat_get_pointer(seat);
            gint root_x, root_y;

            gdk_window_get_root_coords(window,
             *zoom_data->xpos, *zoom_data->ypos, &root_x, &root_y);

            gdk_device_warp(pointer, gdk_display_get_default_screen(display),
                root_x + xo, root_y + yo);
            return TRUE;
        } else {
            *zoom_data->command = (char)gdk_keyval_to_unicode( e->keyval);
            if (e->keyval == GDK_KEY_dead_circumflex)
                *zoom_data->command = '^';
            //printf("%d (%s): %d\n", e->keyval, gdk_keyval_name( e->keyval), *zoom_data->command);
            ggtk_deactivate_zoom( zoom_data);
            return TRUE;
        }
    }
    return FALSE;
}

void ggtk_activate_zoom( gtv_zoom_data_t *data)
{

    if (data->called_from_main_thread)
        return;
#ifndef GAG_USE_GRAB
#ifndef WIN32
    _use_grab_pointer = 1;
    _use_grab_keyboard = 1;
#endif
#endif
    *data->command = 0;
    ggtk_set_pointer_event_handler( _zoom_handler, data);

    ggtk_env_t *genv = (ggtk_env_t *)data->env;
    GdkWindow *win = gtk_widget_get_window(genv->drawing_area);
    GdkDisplay *display = gtk_widget_get_display(genv->drawing_area);
    GdkSeat *seat = gdk_display_get_default_seat(display);

    gtk_widget_add_events(genv->drawing_area, GDK_POINTER_MOTION_MASK);
    if (_use_grab_pointer) {
        gdk_seat_grab(seat, win, GDK_SEAT_CAPABILITY_POINTER, 
            TRUE, NULL, NULL, NULL, NULL);
    }
    if (_use_grab_keyboard) {
        GdkGrabStatus statusKeyboard = gdk_seat_grab(
            seat,win,GDK_SEAT_CAPABILITY_KEYBOARD,
            TRUE,NULL,NULL,NULL,NULL
        );
        if (statusKeyboard != GDK_GRAB_SUCCESS)
            fprintf( stderr, "gdk_keyboard_grab error\n");
    } else {
#ifdef WIN32
        gtk_window_present(gtk_widget_get_window(genv->main_widget));
#else
        gdk_window_focus(gtk_widget_get_window(genv->main_widget), GDK_CURRENT_TIME);
#endif
    }
    ggtk_set_post_refresh_handler( genv, _draw_zoom_cursor_handler, data);

    gint x, y;
    GdkModifierType mask;
    GdkDevice *pointer = gdk_seat_get_pointer(seat);
    gdk_window_get_device_position(win, pointer, &x, &y, &mask);
    _draw_zoom_cursor( data, x, y, _ZOOM_FIRST);

    gtk_widget_set_can_focus(genv->drawing_area, TRUE);
    gtk_widget_grab_focus(genv->drawing_area);
}
