/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.pcode.emu.jit.gen.tgt;

import java.math.BigInteger;
import java.util.*;

import org.objectweb.asm.Opcodes;

import ghidra.pcode.emu.jit.JitCompiler;
import ghidra.pcode.emu.jit.JitPassage.*;
import ghidra.pcode.emu.jit.JitPcodeThread;
import ghidra.pcode.emu.jit.analysis.JitDataFlowUseropLibrary;
import ghidra.pcode.emu.jit.analysis.JitType.IntJitType;
import ghidra.pcode.emu.jit.analysis.JitType.LongJitType;
import ghidra.pcode.emu.jit.gen.JitCodeGenerator;
import ghidra.pcode.exec.*;
import ghidra.pcode.exec.PcodeUseropLibrary.PcodeUseropDefinition;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressFactory;
import ghidra.program.model.lang.*;
import ghidra.program.model.pcode.PcodeOp;
import ghidra.program.model.pcode.Varnode;
import ghidra.program.util.DefaultLanguageService;

/**
 * The interface implemented by classfiles generated by {@link JitCompiler}.
 * 
 * <p>
 * This also serves as a run-time library of routines that implement p-code ops not trivially
 * implemented by the JVM or its run-time library. In theory, they can be inlined by the JVM's JIT
 * at its discretion.
 */
public interface JitCompiledPassage {

	/**
	 * An entry point that is not yet bound to a specific thread
	 * 
	 * @implNote This would be a {@code record} except that it maintains the cache of instances per
	 *           thread
	 */
	class EntryPointPrototype {
		private final JitCompiledPassageClass cls;
		private final int blockId;
		private final Map<JitPcodeThread, EntryPoint> perThread = new HashMap<>();

		/**
		 * Construct an entry prototype
		 * 
		 * @param cls the compiled passage class (i.e., passage not yet bound to a specific thread)
		 * @param blockId the block at which to enter the passage
		 */
		public EntryPointPrototype(JitCompiledPassageClass cls, int blockId) {
			this.cls = cls;
			this.blockId = blockId;
		}

		@Override
		public String toString() {
			return "EntryPointPrototype[%s,%d]".formatted(cls, blockId);
		}

		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof EntryPointPrototype that)) {
				return false;
			}
			if (!this.cls.equals(that.cls)) {
				return false;
			}
			if (this.blockId != that.blockId) {
				return false;
			}
			return true;
		}

		@Override
		public int hashCode() {
			return Objects.hash(cls, blockId);
		}

		/**
		 * Create (or get) the entry point for the given thread
		 * 
		 * @param thread the thread to bind to the entry point
		 * @return the resulting entry point
		 * @see JitPcodeThread#getEntry(AddrCtx)
		 */
		public EntryPoint createInstance(JitPcodeThread thread) {
			synchronized (perThread) {
				return perThread.computeIfAbsent(thread,
					t -> new EntryPoint(this, cls.createInstance(t), blockId));
			}
		}
	}

	/**
	 * An entry point into a translated passage
	 * 
	 * <p>
	 * This represents a translated passage and an index into its list of entry points. For an
	 * overview of how this fits into the JIT-accelerated execution loop, see
	 * {@link JitPcodeThread}, especially the <b>Translate</b> and <b>Execute</b> sections. For
	 * details of how the entry points and their metadata are exported, see
	 * {@link JitCodeGenerator}, especially the <b>Entry Point Dispatch</b> section.
	 * 
	 * @param prototype the entry point prototype (passage class and blockId without bound thread)
	 * @param passage the compiled passage, instantiated for the bound thread
	 * @param blockId an index identifying the block at the target address and contextreg value of
	 *            this entry point
	 */
	record EntryPoint(EntryPointPrototype prototype, JitCompiledPassage passage, int blockId) {
		/**
		 * Start/resume execution of the bound thread at this entry point.
		 * 
		 * <p>
		 * The associated passage is invoked, starting at the given block via
		 * {@link JitCompiledPassage#run(int)}, which was generated by {@link JitCodeGenerator}.
		 * 
		 * @return as in {@link JitCompiledPassage#run(int)}
		 */
		public EntryPoint run() {
			return passage.run(blockId);
		}
	}

	/**
	 * A cache slot for a chained entry point
	 * 
	 * <p>
	 * One of these is constructed for each target of a direct branch that exits the passage,
	 * including those of synthetic {@link ExitPcodeOp exit} ops. For each such branch, the
	 * {@link JitCodeGenerator} emits code to invoke {@link #getChained()} on the target's exit slot
	 * and to return the resulting entry point.
	 */
	public static class ExitSlot {
		private final JitPcodeThread thread;
		private final AddrCtx pcCtx;

		private EntryPoint chained;

		/**
		 * Construct an exit slot for the given target and bound thread
		 * 
		 * @param thread the bound thread for the passage constructing this slot
		 * @param target the offset of the target address
		 * @param ctx the target decode context
		 */
		public ExitSlot(JitPcodeThread thread, long target, RegisterValue ctx) {
			this.thread = thread;
			this.pcCtx =
				new AddrCtx(ctx, thread.getLanguage().getDefaultSpace().getAddress(target));
		}

		/**
		 * Get the entry point for this target
		 * 
		 * <p>
		 * This may cause the emulator to translate a new passage.
		 * 
		 * @return the entry point
		 * @implNote This will always return a non-null entry point, even if the branch target is
		 *           invalid. In that case, the "passage" will consist of a single
		 *           {@link DecodeErrorInstruction}, which will ensure the emulator crashes upon
		 *           trying to execute at the target address.
		 */
		public EntryPoint getChained() {
			if (chained == null) {
				chained = computeChained();
			}
			return chained;
		}

		private EntryPoint computeChained() {
			return thread.getEntry(pcCtx);
		}
	}

	/**
	 * Run the compiled passage of code
	 * 
	 * <p>
	 * Except during testing, this is ordinarily called by {@link EntryPoint#run()}. Too see how
	 * this fits into the overall JIT-accelerated execution loop, see {@link JitPcodeThread}. All
	 * implementations of this interface are generated dynamically. To understand that process and
	 * how the entry points are generated and exported, see {@link JitCompiler}.
	 * 
	 * <p>
	 * This method may or may not return a chained entry point. Each passage caches a chained entry
	 * point for each of its direct branch targets. This averts a map lookup on subsequent exits via
	 * the same branch. If a chained entry point is returned, the thread ought to execute it
	 * immediately, unless it has become suspended. Otherwise, the thread must repeat its execution
	 * loop at the <b>Fetch</b> step.
	 * 
	 * @param blockId an index identifying the target address and contextreg where execution should
	 *            enter
	 * @return a chained entry point, or {@code null}
	 */
	EntryPoint run(int blockId);

	/**
	 * Read an {@link IntJitType#I1 int1} from the given array at the given offset
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readInt1(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]);
	}

	/**
	 * Read an {@link IntJitType#I2 int2} from the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntBE2(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) << 8 |
			Byte.toUnsignedInt(arr[offset + 1]);
	}

	/**
	 * Read an {@link IntJitType#I3 int3} from the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntBE3(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) << 16 |
			Byte.toUnsignedInt(arr[offset + 1]) << 8 |
			Byte.toUnsignedInt(arr[offset + 2]);
	}

	/**
	 * Read an {@link IntJitType#I4 int4} from the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntBE4(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) << 24 |
			Byte.toUnsignedInt(arr[offset + 1]) << 16 |
			Byte.toUnsignedInt(arr[offset + 2]) << 8 |
			Byte.toUnsignedInt(arr[offset + 3]);
	}

	/**
	 * Read an {@link IntJitType#I2 int2} from the given array at the given offset in little endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntLE2(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) |
			Byte.toUnsignedInt(arr[offset + 1]) << 8;
	}

	/**
	 * Read an {@link IntJitType#I3 int3} from the given array at the given offset in little endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntLE3(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) |
			Byte.toUnsignedInt(arr[offset + 1]) << 8 |
			Byte.toUnsignedInt(arr[offset + 2]) << 16;
	}

	/**
	 * Read an {@link IntJitType#I4 int4} from the given array at the given offset in little endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM int
	 */
	static int readIntLE4(byte[] arr, int offset) {
		return Byte.toUnsignedInt(arr[offset]) |
			Byte.toUnsignedInt(arr[offset + 1]) << 8 |
			Byte.toUnsignedInt(arr[offset + 2]) << 16 |
			Byte.toUnsignedInt(arr[offset + 3]) << 24;
	}

	/**
	 * Write an {@link IntJitType#I1 int1} into the given array at the given offset
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeInt1(int value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I2 int2} into the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntBE2(int value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 8);
		arr[offset + 1] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I3 int3} into the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntBE3(int value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 16);
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I4 int4} into the given array at the given offset in big endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntBE4(int value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 24);
		arr[offset + 1] = (byte) (value >> 16);
		arr[offset + 2] = (byte) (value >> 8);
		arr[offset + 3] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I2 int2} into the given array at the given offset in litte endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntLE2(int value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
	}

	/**
	 * Write an {@link IntJitType#I3 int3} into the given array at the given offset in litte endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntLE3(int value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
	}

	/**
	 * Write an {@link IntJitType#I4 int4} into the given array at the given offset in litte endian
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM int
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeIntLE4(int value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
	}

	/**
	 * Read an {@link IntJitType#I1 int1} from the given array at the given offset.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I1 int1} can fit in
	 * a JVM int, this method is used when reading 1 byte of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLong1(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]);
	}

	/**
	 * Read an {@link IntJitType#I2 int2} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I2 int2} can fit in
	 * a JVM int, this method is used when reading 2 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE2(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 8 |
			Byte.toUnsignedLong(arr[offset + 1]);
	}

	/**
	 * Read an {@link IntJitType#I3 int3} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I3 int3} can fit in
	 * a JVM int, this method is used when reading 3 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE3(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 16 |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]);
	}

	/**
	 * Read an {@link IntJitType#I4 int4} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I4 int4} can fit in
	 * a JVM int, this method is used when reading 4 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE4(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 24 |
			Byte.toUnsignedLong(arr[offset + 1]) << 16 |
			Byte.toUnsignedLong(arr[offset + 2]) << 8 |
			Byte.toUnsignedLong(arr[offset + 3]);
	}

	/**
	 * Read an {@link LongJitType#I5 int5} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE5(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 32 |
			Byte.toUnsignedLong(arr[offset + 1]) << 24 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 8 |
			Byte.toUnsignedLong(arr[offset + 4]);
	}

	/**
	 * Read an {@link LongJitType#I6 int6} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE6(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 40 |
			Byte.toUnsignedLong(arr[offset + 1]) << 32 |
			Byte.toUnsignedLong(arr[offset + 2]) << 24 |
			Byte.toUnsignedLong(arr[offset + 3]) << 16 |
			Byte.toUnsignedLong(arr[offset + 4]) << 8 |
			Byte.toUnsignedLong(arr[offset + 5]);
	}

	/**
	 * Read an {@link LongJitType#I7 int7} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE7(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 48 |
			Byte.toUnsignedLong(arr[offset + 1]) << 40 |
			Byte.toUnsignedLong(arr[offset + 2]) << 32 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24 |
			Byte.toUnsignedLong(arr[offset + 4]) << 16 |
			Byte.toUnsignedLong(arr[offset + 5]) << 8 |
			Byte.toUnsignedLong(arr[offset + 6]);
	}

	/**
	 * Read an {@link LongJitType#I8 int8} from the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongBE8(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) << 56 |
			Byte.toUnsignedLong(arr[offset + 1]) << 48 |
			Byte.toUnsignedLong(arr[offset + 2]) << 40 |
			Byte.toUnsignedLong(arr[offset + 3]) << 32 |
			Byte.toUnsignedLong(arr[offset + 4]) << 24 |
			Byte.toUnsignedLong(arr[offset + 5]) << 16 |
			Byte.toUnsignedLong(arr[offset + 6]) << 8 |
			Byte.toUnsignedLong(arr[offset + 7]);
	}

	/**
	 * Read an {@link IntJitType#I2 int2} from the given array at the given offset in little endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I2 int2} can fit in
	 * a JVM int, this method is used when reading 2 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE2(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8;
	}

	/**
	 * Read an {@link IntJitType#I3 int3} from the given array at the given offset in little endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I3 int3} can fit in
	 * a JVM int, this method is used when reading 3 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE3(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16;
	}

	/**
	 * Read an {@link IntJitType#I4 int4} from the given array at the given offset in little endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I4 int4} can fit in
	 * a JVM int, this method is used when reading 4 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE4(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24;
	}

	/**
	 * Read an {@link LongJitType#I5 int5} from the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE5(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24 |
			Byte.toUnsignedLong(arr[offset + 4]) << 32;
	}

	/**
	 * Read an {@link LongJitType#I6 int6} from the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE6(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24 |
			Byte.toUnsignedLong(arr[offset + 4]) << 32 |
			Byte.toUnsignedLong(arr[offset + 5]) << 40;
	}

	/**
	 * Read an {@link LongJitType#I7 int7} from the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE7(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24 |
			Byte.toUnsignedLong(arr[offset + 4]) << 32 |
			Byte.toUnsignedLong(arr[offset + 5]) << 40 |
			Byte.toUnsignedLong(arr[offset + 6]) << 48;
	}

	/**
	 * Read an {@link LongJitType#I8 int8} from the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param arr the array
	 * @param offset the offset
	 * @return the value as a JVM long
	 */
	static long readLongLE8(byte[] arr, int offset) {
		return Byte.toUnsignedLong(arr[offset]) |
			Byte.toUnsignedLong(arr[offset + 1]) << 8 |
			Byte.toUnsignedLong(arr[offset + 2]) << 16 |
			Byte.toUnsignedLong(arr[offset + 3]) << 24 |
			Byte.toUnsignedLong(arr[offset + 4]) << 32 |
			Byte.toUnsignedLong(arr[offset + 5]) << 40 |
			Byte.toUnsignedLong(arr[offset + 6]) << 48 |
			Byte.toUnsignedLong(arr[offset + 7]) << 56;
	}

	/**
	 * Write an {@link IntJitType#I1 int1} into the given array at the given offset.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I1 int1} can fit in
	 * a JVM int, this method is used when writing 1 byte of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLong1(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I2 int2} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I2 int2} can fit in
	 * a JVM int, this method is used when writing 2 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE2(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 8);
		arr[offset + 1] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I3 int3} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I3 int3} can fit in
	 * a JVM int, this method is used when writing 3 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE3(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 16);
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I4 int4} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I4 int4} can fit in
	 * a JVM int, this method is used when writing 4 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE4(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 24);
		arr[offset + 1] = (byte) (value >> 16);
		arr[offset + 2] = (byte) (value >> 8);
		arr[offset + 3] = (byte) value;
	}

	/**
	 * Write an {@link LongJitType#I5 int5} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE5(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 32);
		arr[offset + 1] = (byte) (value >> 24);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 8);
		arr[offset + 4] = (byte) value;
	}

	/**
	 * Write an {@link LongJitType#I6 int6} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE6(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 40);
		arr[offset + 1] = (byte) (value >> 32);
		arr[offset + 2] = (byte) (value >> 24);
		arr[offset + 3] = (byte) (value >> 16);
		arr[offset + 4] = (byte) (value >> 8);
		arr[offset + 5] = (byte) value;
	}

	/**
	 * Write an {@link LongJitType#I7 int7} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE7(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 48);
		arr[offset + 1] = (byte) (value >> 40);
		arr[offset + 2] = (byte) (value >> 32);
		arr[offset + 3] = (byte) (value >> 24);
		arr[offset + 4] = (byte) (value >> 16);
		arr[offset + 5] = (byte) (value >> 8);
		arr[offset + 6] = (byte) value;
	}

	/**
	 * Write an {@link LongJitType#I8 int8} into the given array at the given offset in big endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongBE8(long value, byte[] arr, int offset) {
		arr[offset] = (byte) (value >> 56);
		arr[offset + 1] = (byte) (value >> 48);
		arr[offset + 2] = (byte) (value >> 40);
		arr[offset + 3] = (byte) (value >> 32);
		arr[offset + 4] = (byte) (value >> 24);
		arr[offset + 5] = (byte) (value >> 16);
		arr[offset + 6] = (byte) (value >> 8);
		arr[offset + 7] = (byte) value;
	}

	/**
	 * Write an {@link IntJitType#I2 int2} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I2 int2} can fit in
	 * a JVM int, this method is used when writing 2 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE2(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
	}

	/**
	 * Write an {@link IntJitType#I3 int3} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I3 int3} can fit in
	 * a JVM int, this method is used when writing 3 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE3(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
	}

	/**
	 * Write an {@link IntJitType#I4 int4} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code. While an {@link IntJitType#I4 int4} can fit in
	 * a JVM int, this method is used when writing 4 bytes of a {@link LongJitType larger int} that
	 * spans a page boundary.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE4(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
	}

	/**
	 * Write an {@link LongJitType#I5 int5} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE5(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
		arr[offset + 4] = (byte) (value >> 32);
	}

	/**
	 * Write an {@link LongJitType#I6 int6} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE6(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
		arr[offset + 4] = (byte) (value >> 32);
		arr[offset + 5] = (byte) (value >> 40);
	}

	/**
	 * Write an {@link LongJitType#I7 int7} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE7(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
		arr[offset + 4] = (byte) (value >> 32);
		arr[offset + 5] = (byte) (value >> 40);
		arr[offset + 6] = (byte) (value >> 48);
	}

	/**
	 * Write an {@link LongJitType#I8 int8} into the given array at the given offset in little
	 * endian.
	 * 
	 * <p>
	 * This is invoked by dynamically generated code.
	 * 
	 * @param value the value as a JVM long
	 * @param arr the array
	 * @param offset the offset
	 */
	static void writeLongLE8(long value, byte[] arr, int offset) {
		arr[offset] = (byte) value;
		arr[offset + 1] = (byte) (value >> 8);
		arr[offset + 2] = (byte) (value >> 16);
		arr[offset + 3] = (byte) (value >> 24);
		arr[offset + 4] = (byte) (value >> 32);
		arr[offset + 5] = (byte) (value >> 40);
		arr[offset + 6] = (byte) (value >> 48);
		arr[offset + 7] = (byte) (value >> 56);
	}

	/**
	 * Convert two integers into a single long
	 * 
	 * <p>
	 * In terms of the JVM stack, this simply converts the top two ints to an equivalent long.
	 * <b>TODO</b>: This no longer appears to be used, but may be in anticipation of multi-precision
	 * integer support.
	 * 
	 * @param msl the more significant leg
	 * @param lsl the less significant leg
	 * @return the long
	 */
	static long conv2IntToLong(int msl, int lsl) {
		return Integer.toUnsignedLong(msl) << Integer.SIZE | Integer.toUnsignedLong(lsl);
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SBORROW int_sborrow} on JVM ints.
	 * 
	 * <p>
	 * This actually computes all the borrow bits. To extract a specific one, the generator should
	 * emit a shift and mask.
	 * 
	 * @param a the first operand as in {@code a - b}
	 * @param b the second operand as in {@code a - b}
	 * @return the register of borrow bits
	 */
	static int sBorrowIntRaw(int a, int b) {
		int r = a - b;
		a ^= r;
		r ^= b;
		r ^= -1;
		a &= r;
		return a;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SBORROW int_sborrow} on JVM longs.
	 * 
	 * <p>
	 * This actually computes all the borrow bits. To extract a specific one, the generator should
	 * emit a shift and mask.
	 * 
	 * @param a the first operand as in {@code a - b}
	 * @param b the second operand as in {@code a - b}
	 * @return the register of borrow bits
	 */
	static long sBorrowLongRaw(long a, long b) {
		long r = a - b;
		a ^= r;
		r ^= b;
		r ^= -1;
		a &= r;
		return a;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SCARRY int_scarry} on JVM ints.
	 * 
	 * <p>
	 * This actually computes all the carry bits. To extract a specific one, the generator should
	 * emit a shift and mask.
	 * 
	 * @param a the first operand as in {@code a + b}
	 * @param b the second operand as in {@code a + b}
	 * @return the register of carry bits
	 */
	static int sCarryIntRaw(int a, int b) {
		int r = a + b;
		r ^= a;
		a ^= b;
		a ^= -1;
		r &= a;
		return r;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SCARRY int_scarry} on JVM longs.
	 * 
	 * <p>
	 * This actually computes all the carry bits. To extract a specific one, the generator should
	 * emit a shift and mask.
	 * 
	 * @param a the first operand as in {@code a + b}
	 * @param b the second operand as in {@code a + b}
	 * @return the register of carry bits
	 */
	static long sCarryLongRaw(long a, long b) {
		long r = a + b;
		r ^= a;
		a ^= b;
		a ^= -1;
		r &= a;
		return r;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_LEFT int_left} on JVM longs.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHL ishl}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val << amt}
	 * @param amt the amt as in {@code val << amt}
	 * @return the value
	 */
	static long intLeft(long val, long amt) {
		if (Long.compareUnsigned(amt, Long.SIZE) >= 0) {
			return 0;
		}
		return val << amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_LEFT int_left} on JVM long with int amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHL ishl}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val << amt}
	 * @param amt the amt as in {@code val << amt}
	 * @return the value
	 */
	static long intLeft(long val, int amt) {
		if (Integer.compareUnsigned(amt, Long.SIZE) >= 0) {
			return 0;
		}
		return val << amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_LEFT int_left} on JVM int with long amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHL ishl}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val << amt}
	 * @param amt the amt as in {@code val << amt}
	 * @return the value
	 */
	static int intLeft(int val, long amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return 0;
		}
		return val << amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_LEFT int_left} on JVM ints.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHL ishl}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val << amt}
	 * @param amt the amt as in {@code val << amt}
	 * @return the value
	 */
	static int intLeft(int val, int amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return 0;
		}
		return val << amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_RIGHT int_right} on JVM longs.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#IUSHR iushr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val >> amt}
	 * @param amt the amt as in {@code val >> amt}
	 * @return the value
	 */
	static long intRight(long val, long amt) {
		if (Long.compareUnsigned(amt, Long.SIZE) >= 0) {
			return 0;
		}
		return val >>> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_RIGHT int_right} on JVM long with int amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#IUSHR iushr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val >> amt}
	 * @param amt the amt as in {@code val >> amt}
	 * @return the value
	 */
	static long intRight(long val, int amt) {
		if (Integer.compareUnsigned(amt, Long.SIZE) >= 0) {
			return 0;
		}
		return val >>> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_RIGHT int_right} on JVM int with long amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#IUSHR iushr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val >> amt}
	 * @param amt the amt as in {@code val >> amt}
	 * @return the value
	 */
	static int intRight(int val, long amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return 0;
		}
		return val >>> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_RIGHT int_right} on JVM ints.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#IUSHR iushr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size clear the register.
	 * 
	 * @param val the value as in {@code val >> amt}
	 * @param amt the amt as in {@code val >> amt}
	 * @return the value
	 */
	static int intRight(int val, int amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return 0;
		}
		return val >>> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SRIGHT int_sright} on JVM longs.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHR ishr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size fill the register with
	 * the sign bit.
	 * 
	 * @param val the value as in {@code val s>> amt}
	 * @param amt the amt as in {@code val s>> amt}
	 * @return the value
	 */
	static long intSRight(long val, long amt) {
		if (Long.compareUnsigned(amt, Long.SIZE) >= 0) {
			return val >> (Long.SIZE - 1);
		}
		return val >> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SRIGHT int_sright} on JVM long with int amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHR ishr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size fill the register with
	 * the sign bit.
	 * 
	 * @param val the value as in {@code val s>> amt}
	 * @param amt the amt as in {@code val s>> amt}
	 * @return the value
	 */
	static long intSRight(long val, int amt) {
		if (Integer.compareUnsigned(amt, Long.SIZE) >= 0) {
			return val >> (Long.SIZE - 1);
		}
		return val >> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SRIGHT int_sright} on JVM int with long amt.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHR ishr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size fill the register with
	 * the sign bit.
	 * 
	 * @param val the value as in {@code val s>> amt}
	 * @param amt the amt as in {@code val s>> amt}
	 * @return the value
	 */
	static int intSRight(int val, long amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return val >> (Integer.SIZE - 1);
		}
		return val >> amt;
	}

	/**
	 * The implementation of {@link PcodeOp#INT_SRIGHT int_sright} on JVM ints.
	 * 
	 * <p>
	 * The semantics here are subtly different than the JVM's {@link Opcodes#ISHR ishr}: 1) The
	 * amount must be treated as unsigned. 2) Shifts in excess of val's size fill the register with
	 * the sign bit.
	 * 
	 * @param val the value as in {@code val s>> amt}
	 * @param amt the amt as in {@code val s>> amt}
	 * @return the value
	 */
	static int intSRight(int val, int amt) {
		if (Long.compareUnsigned(amt, Integer.SIZE) >= 0) {
			return val >> (Integer.SIZE - 1);
		}
		return val >> amt;
	}

	/**
	 * Get the language for the given string language ID
	 * 
	 * <p>
	 * This is called by generated static initializers.
	 * 
	 * @param languageID the language ID
	 * @return the language
	 * @throws LanguageNotFoundException if the language is not found
	 */
	static Language getLanguage(String languageID) throws LanguageNotFoundException {
		return DefaultLanguageService.getLanguageService().getLanguage(new LanguageID(languageID));
	}

	/**
	 * Construct a contextreg value from the given language and hex value
	 * 
	 * <p>
	 * This is called by generated static initializers to pre-construct context values.
	 * 
	 * @param language the language
	 * @param value the value as a string of hexadecimal digits
	 * @return the value
	 */
	static RegisterValue createContext(Language language, String value) {
		return new RegisterValue(language.getContextBaseRegister(),
			new BigInteger(value, 16));
	}

	/**
	 * Construct a varnode
	 * 
	 * <p>
	 * This is called by generated static initializers to pre-construct any varnodes it needs to
	 * re-create, mostly for invoking userops with the Standard strategy.
	 * 
	 * @param factory the language's address factory
	 * @param space the name of the space
	 * @param offset the byte offset
	 * @param size the size (in bytes)
	 * @return the varnode
	 */
	static Varnode createVarnode(AddressFactory factory, String space, long offset, int size) {
		return new Varnode(factory.getAddressSpace(space).getAddress(offset), size);
	}

	/**
	 * Get this instance's bound thread.
	 * 
	 * <p>
	 * The generator implements a standard getter. This is frequently used by other default methods
	 * of this interface, which are in turn invoked by generated code.
	 * 
	 * @return the thread
	 */
	JitPcodeThread thread();

	/**
	 * Set the bound thread's program counter and decode context.
	 * 
	 * <p>
	 * This is called during retirement, i.e., upon exiting a passage. This just converts things to
	 * the right type and invokes
	 * {@link JitPcodeThread#writeCounterAndContext(Address, RegisterValue)}.
	 * 
	 * @param counter the offset of the next instruction to execute
	 * @param context the decode context for the next instruction
	 */
	default void writeCounterAndContext(long counter, RegisterValue context) {
		JitPcodeThread thread = thread();
		Address pc = thread.getLanguage().getDefaultSpace().getAddress(counter);
		thread.writeCounterAndContext(pc, context);
	}

	/**
	 * Set the bound thread's program counter and decode context, without writing it to the machine
	 * state.
	 * 
	 * <p>
	 * This is called during retirement upon entering a hazard. This just converts things to the
	 * right type and invokes {@link JitPcodeThread#setCounterAndContext(Address, RegisterValue)}.
	 * 
	 * @param counter the offset of the next instruction to execute
	 * @param context the decode context for the next instruction
	 */
	default void setCounterAndContext(long counter, RegisterValue context) {
		JitPcodeThread thread = thread();
		Address pc = thread.getLanguage().getDefaultSpace().getAddress(counter);
		thread.setCounterAndContext(pc, context);
	}

	/**
	 * Retrieve a userop definition from the bound thread.
	 * 
	 * <p>
	 * This is invoked by generated constructors to retain a userop reference for later invocation.
	 * Note that it is the userop as defined by the user or emulator, not any wrapper used during
	 * decode or translation. Depending on the invocation strategy, this reference may be saved and
	 * later used with {@link #invokeUserop(PcodeUseropDefinition, Varnode, Varnode[])}, or its
	 * method and instance may be extracted and saved for Direct invocation later.
	 * 
	 * @param name the name of the userop
	 * @return the userop or {@code null}
	 * @see JitDataFlowUseropLibrary
	 */
	default PcodeUseropDefinition<byte[]> getUseropDefinition(String name) {
		return thread().getUseropLibrary().getUserops().get(name);
	}

	/**
	 * Invoke the given userop on the bound thread with the given operands
	 * 
	 * <p>
	 * This is invoked by generated code in {@link JitCompiledPassage#run(int)} to invoke a userop
	 * via the Standard strategy.
	 * 
	 * @param userop the userop definition
	 * @param output an optional output operand
	 * @param inputs the input operands
	 * @see JitDataFlowUseropLibrary
	 * @see PcodeUseropDefinition#execute(PcodeExecutor, PcodeUseropLibrary, Varnode, List)
	 */
	default void invokeUserop(PcodeUseropDefinition<byte[]> userop, Varnode output,
			Varnode[] inputs) {
		userop.execute(thread().getExecutor(), thread().getUseropLibrary(), output,
			Arrays.asList(inputs));
	}

	/**
	 * Construct an exception when attempting to execute an "instruction" that could not be decoded.
	 * 
	 * <p>
	 * When the decoder encounters an error, instead of crashing immediately, it must consider that
	 * execution may not actually reach the error, so it instead emits pseudo-instructions
	 * describing the error. The translator then emits code that will invoke this method and throw
	 * the result. Thus, we only crash if the erroneous condition is actually met.
	 * 
	 * @param message the human-readable message
	 * @param counter the program counter where the decode error was encountered
	 * @return the exception, which should be thrown immediately
	 */
	default DecodePcodeExecutionException createDecodeError(String message, long counter) {
		return new DecodePcodeExecutionException(message,
			thread().getLanguage().getDefaultSpace().getAddress(counter));
	}

	/**
	 * Construct an exit slot for the given branch target
	 * 
	 * <p>
	 * This is invoked by generated constructors for each branch target that exits the passage. Each
	 * is saved as a field and will be filled lazily with its chained entry point the first time the
	 * branch is taken.
	 * 
	 * @param target the target program counter
	 * @param ctx the target decode context
	 * @return the exit slot
	 */
	default ExitSlot createExitSlot(long target, RegisterValue ctx) {
		return new ExitSlot(thread(), target, ctx);
	}

	/**
	 * Get the chained entry point for the given exit point's target
	 * 
	 * <p>
	 * This is invoked by generated code in {@link JitCompiledPassage#run(int)} to take a branch
	 * exiting the passage. The first time, the exit slot is lazily filled, possibly requiring
	 * further JIT translation.
	 * 
	 * @param slot the slot for the target of the branch we're taking
	 * @return the chained entry point
	 */
	static EntryPoint getChained(ExitSlot slot) {
		return slot.getChained();
	}

	/**
	 * Invoke {@link JitPcodeThread#count(int, int)} for the bound thread
	 * 
	 * @param instructions as in {@link JitPcodeThread#count(int, int)}
	 * @param trailingOps as in {@link JitPcodeThread#count(int, int)}
	 */
	default void count(int instructions, int trailingOps) {
		thread().count(instructions, trailingOps);
	}
}
