import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { render } from '../../test-utils/render.js';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { DefaultAppLayout } from './DefaultAppLayout.js';
import { StreamingState } from '../types.js';
import { Text } from 'ink';
// Mock dependencies
const mockUIState = {
    rootUiRef: { current: null },
    terminalHeight: 24,
    terminalWidth: 80,
    mainAreaWidth: 80,
    backgroundShells: new Map(),
    activeBackgroundShellPid: null,
    backgroundShellHeight: 10,
    embeddedShellFocused: false,
    dialogsVisible: false,
    streamingState: StreamingState.Idle,
    isBackgroundShellListOpen: false,
    mainControlsRef: { current: null },
    customDialog: null,
    historyManager: { addItem: vi.fn() },
    history: [],
    pendingHistoryItems: [],
    slashCommands: [],
    constrainHeight: false,
    availableTerminalHeight: 20,
    activePtyId: null,
    isBackgroundShellVisible: true,
};
vi.mock('../contexts/UIStateContext.js', () => ({
    useUIState: () => mockUIState,
}));
vi.mock('../hooks/useFlickerDetector.js', () => ({
    useFlickerDetector: vi.fn(),
}));
vi.mock('../hooks/useAlternateBuffer.js', () => ({
    useAlternateBuffer: vi.fn(() => false),
}));
vi.mock('../contexts/ConfigContext.js', () => ({
    useConfig: () => ({
        getAccessibility: vi.fn(() => ({
            enableLoadingPhrases: true,
        })),
    }),
}));
// Mock child components to simplify output
vi.mock('../components/LoadingIndicator.js', () => ({
    LoadingIndicator: () => _jsx(Text, { children: "LoadingIndicator" }),
}));
vi.mock('../components/MainContent.js', () => ({
    MainContent: () => _jsx(Text, { children: "MainContent" }),
}));
vi.mock('../components/Notifications.js', () => ({
    Notifications: () => _jsx(Text, { children: "Notifications" }),
}));
vi.mock('../components/DialogManager.js', () => ({
    DialogManager: () => _jsx(Text, { children: "DialogManager" }),
}));
vi.mock('../components/Composer.js', () => ({
    Composer: () => _jsx(Text, { children: "Composer" }),
}));
vi.mock('../components/ExitWarning.js', () => ({
    ExitWarning: () => _jsx(Text, { children: "ExitWarning" }),
}));
vi.mock('../components/CopyModeWarning.js', () => ({
    CopyModeWarning: () => _jsx(Text, { children: "CopyModeWarning" }),
}));
vi.mock('../components/BackgroundShellDisplay.js', () => ({
    BackgroundShellDisplay: () => _jsx(Text, { children: "BackgroundShellDisplay" }),
}));
const createMockShell = (pid) => ({
    pid,
    command: 'test command',
    output: 'test output',
    isBinary: false,
    binaryBytesReceived: 0,
    status: 'running',
});
describe('<DefaultAppLayout />', () => {
    beforeEach(() => {
        vi.clearAllMocks();
        // Reset mock state defaults
        mockUIState.backgroundShells = new Map();
        mockUIState.activeBackgroundShellPid = null;
        mockUIState.streamingState = StreamingState.Idle;
    });
    it('renders BackgroundShellDisplay when shells exist and active', async () => {
        mockUIState.backgroundShells.set(123, createMockShell(123));
        mockUIState.activeBackgroundShellPid = 123;
        mockUIState.backgroundShellHeight = 5;
        const { lastFrame, waitUntilReady, unmount } = render(_jsx(DefaultAppLayout, {}));
        await waitUntilReady();
        expect(lastFrame()).toMatchSnapshot();
        unmount();
    });
    it('hides BackgroundShellDisplay when StreamingState is WaitingForConfirmation', async () => {
        mockUIState.backgroundShells.set(123, createMockShell(123));
        mockUIState.activeBackgroundShellPid = 123;
        mockUIState.backgroundShellHeight = 5;
        mockUIState.streamingState = StreamingState.WaitingForConfirmation;
        const { lastFrame, waitUntilReady, unmount } = render(_jsx(DefaultAppLayout, {}));
        await waitUntilReady();
        expect(lastFrame()).toMatchSnapshot();
        unmount();
    });
    it('shows BackgroundShellDisplay when StreamingState is NOT WaitingForConfirmation', async () => {
        mockUIState.backgroundShells.set(123, createMockShell(123));
        mockUIState.activeBackgroundShellPid = 123;
        mockUIState.backgroundShellHeight = 5;
        mockUIState.streamingState = StreamingState.Responding;
        const { lastFrame, waitUntilReady, unmount } = render(_jsx(DefaultAppLayout, {}));
        await waitUntilReady();
        expect(lastFrame()).toMatchSnapshot();
        unmount();
    });
});
//# sourceMappingURL=DefaultAppLayout.test.js.map