/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useCallback, useRef, useEffect } from 'react';
/**
 * A hook to manage a state value that automatically resets to null after a duration.
 * Useful for transient UI messages, hints, or warnings.
 */
export function useTimedMessage(durationMs) {
    const [message, setMessage] = useState(null);
    const timeoutRef = useRef(null);
    const showMessage = useCallback((msg) => {
        setMessage(msg);
        if (timeoutRef.current) {
            clearTimeout(timeoutRef.current);
        }
        timeoutRef.current = setTimeout(() => {
            setMessage(null);
        }, durationMs);
    }, [durationMs]);
    useEffect(() => () => {
        if (timeoutRef.current) {
            clearTimeout(timeoutRef.current);
        }
    }, []);
    return [message, showMessage];
}
//# sourceMappingURL=useTimedMessage.js.map