/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { debugLogger, } from '@google/gemini-cli-core';
import { ToolCallStatus, } from '../types.js';
export function mapCoreStatusToDisplayStatus(coreStatus) {
    switch (coreStatus) {
        case 'validating':
            return ToolCallStatus.Executing;
        case 'awaiting_approval':
            return ToolCallStatus.Confirming;
        case 'executing':
            return ToolCallStatus.Executing;
        case 'success':
            return ToolCallStatus.Success;
        case 'cancelled':
            return ToolCallStatus.Canceled;
        case 'error':
            return ToolCallStatus.Error;
        case 'scheduled':
            return ToolCallStatus.Pending;
        default:
            debugLogger.warn(`Unknown core status encountered: ${coreStatus}`);
            return ToolCallStatus.Error;
    }
}
/**
 * Transforms `ToolCall` objects into `HistoryItemToolGroup` objects for UI
 * display. This is a pure projection layer and does not track interaction
 * state.
 */
export function mapToDisplay(toolOrTools) {
    const toolCalls = Array.isArray(toolOrTools) ? toolOrTools : [toolOrTools];
    const toolDisplays = toolCalls.map((call) => {
        let description;
        let renderOutputAsMarkdown = false;
        const displayName = call.tool?.displayName ?? call.request.name;
        if (call.status === 'error') {
            description = JSON.stringify(call.request.args);
        }
        else {
            description = call.invocation.getDescription();
            renderOutputAsMarkdown = call.tool.isOutputMarkdown;
        }
        const baseDisplayProperties = {
            callId: call.request.callId,
            name: displayName,
            description,
            renderOutputAsMarkdown,
        };
        let resultDisplay = undefined;
        let confirmationDetails = undefined;
        let outputFile = undefined;
        let ptyId = undefined;
        switch (call.status) {
            case 'success':
                resultDisplay = call.response.resultDisplay;
                outputFile = call.response.outputFile;
                break;
            case 'error':
            case 'cancelled':
                resultDisplay = call.response.resultDisplay;
                break;
            case 'awaiting_approval':
                // Only map if it's the legacy callback-based details.
                // Serializable details will be handled in a later milestone.
                if (call.confirmationDetails &&
                    'onConfirm' in call.confirmationDetails &&
                    typeof call.confirmationDetails.onConfirm === 'function') {
                    confirmationDetails =
                        call.confirmationDetails;
                }
                break;
            case 'executing':
                resultDisplay = call.liveOutput;
                ptyId = call.pid;
                break;
            case 'scheduled':
            case 'validating':
                break;
            default: {
                const exhaustiveCheck = call;
                debugLogger.warn(`Unhandled tool call status in mapper: ${exhaustiveCheck.status}`);
                break;
            }
        }
        return {
            ...baseDisplayProperties,
            status: mapCoreStatusToDisplayStatus(call.status),
            resultDisplay,
            confirmationDetails,
            outputFile,
            ptyId,
        };
    });
    return {
        type: 'tool_group',
        tools: toolDisplays,
    };
}
//# sourceMappingURL=toolMapping.js.map