note
	description	: "System's root class. Initial version automatically generated"
	legal: "See notice at end of class."
	status: "See notice at end of class."

class
	KG_ENGINE

inherit
	ANY

	KG_CONSTANTS
		export
			{NONE} all
		end

	KG_ENVIRONMENT_VARIABLES_IMP
		export
			{NONE} all
		end

create
	make

feature {NONE} -- Initialization

	make (l_product, l_version: STRING)
			-- Set `product' with `l_product'.
			-- Set `version' with `l_version'.
		require
			non_void_product_name: l_product /= Void
			valid_product_name: not l_product.is_empty
			non_void_version: l_version /= Void
			valid_version: not l_version.is_empty
		do
			product := l_product
			version := l_version
		ensure
			product_set: product = l_product
			version_set: version = l_version
		end

feature -- Access

	product: STRING
			-- Product for which to check license info

	version: STRING
			-- Product version number for which to check license info

	username: STRING
			-- Username for product `product' version `version'.
		do
			Result := user_username (product, version)
			if Result.is_empty then
				Result := global_username (product, version)
			end
		end

	cd_key: STRING
			-- CD key for product `product' version `version'.
		do
			Result := user_cd_key (product, version)
			if Result.is_empty then
				Result := global_cd_key (product, version)
			end
		end

feature -- Status Report

	check_license
			-- Check if sofware is correctly activated.
			-- set `can_run'.
		local
			pf: KG_PRIVATE_INFO
		do
			can_run := False
			if is_initialized then
				can_run := is_activated
				if not can_run then
					create {KG_PRIVATE_INFO_IMP} pf.retrieve (product, version)
					if pf.is_consistent then
						can_run := pf.remaining_executions > 0
						if can_run then
							pf.set_remaining_executions (pf.remaining_executions - 1)
						end
					end
				end
			end
		end

	can_run: BOOLEAN
			-- Can protected software be run for user `username' with key `cd_key'?
			-- Call `check_license' to set value

	is_initialized: BOOLEAN
			-- Was software protection scheme initialized?
		do
			Result := username /= Void and then not username.is_empty and then cd_key /= Void and then
				cd_key.count = Cd_key_length and then has_valid_cd_key_format (cd_key)
		end

	is_activated: BOOLEAN
			-- Was software activated for user `username' with key `cd_key'?
		require
			initialized: is_initialized
		local
			pf: KG_PRIVATE_INFO
			ukeygen: KG_ACTIVATION_KEY_GENERATOR
		do
			create {KG_PRIVATE_INFO_IMP} pf.retrieve (product, version)
			if pf.is_consistent then
				if (create {KG_CD_KEY_GENERATOR}.make (product, version)).is_valid_cd_key (cd_key) then
					create ukeygen.make (pf.registration_date, product, version, username, cd_key, platform)
					Result := ukeygen.is_valid_activation_key (pf.activation_key)
				end
			end
		end

	remaining_executions: INTEGER
			-- Amount of times software can be run before it needs to be activated
		local
			pf: KG_PRIVATE_INFO
		do
			create {KG_PRIVATE_INFO_IMP} pf.retrieve (product, version)
			if pf.is_consistent then
				Result := pf.remaining_executions
			end
		end

feature -- Basic Operations

	activate (key: STRING; registration_date: DATE)
			-- Register software for `username' with cd key `cd_key'
			-- and activation key `key' at `registration_date'.
			-- Set `is_activated' to `True' if successful.
		require
			non_void_date: registration_date /= Void
			valid_date: registration_date > (create {DATE}.make (2002, 08, 06))
		local
			pf: KG_PRIVATE_INFO
			utc, now: DATE
		do
			create {KG_PRIVATE_INFO_IMP} pf.retrieve (product, version)
			create utc.make_now_utc
			create now.make_now
			now := registration_date + (create {DATE_DURATION}.make_by_days (utc.days - now.days))
			pf.set_key (key, now)
		end

	initialize (name, key: READABLE_STRING_GENERAL)
			-- Initialize software protection scheme for user `name' with
			-- cd key `key'.
		require
			non_void_username: name /= Void
			valid_username: not name.is_empty and then name.is_valid_as_string_8
			non_void_key: key /= Void
			valid_key: not key.is_empty and then key.is_valid_as_string_8
		local
			pf: KG_PRIVATE_INFO
			l_name, l_key: STRING_8
		do
			create l_name.make (name.count)
			l_name.append_string_general (name)
			create l_key.make (key.count)
			l_key.append_string_general (key)
			set_global_username (l_name, product, version)
			set_global_cd_key (l_key, product, version)
			set_user_username (l_name, product, version)
			set_user_cd_key (l_key, product, version)
			create {KG_PRIVATE_INFO_IMP} pf.initialize (product, version)
		ensure
			username_set: username.same_string_general (name)
			cd_key_set: cd_key.same_string_general (key)
		end

note
	copyright:	"Copyright (c) 1984-2006, Eiffel Software and others"
	license:	"Eiffel Forum License v2 (see http://www.eiffel.com/licensing/forum.txt)"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"




end -- class KG_ENGINE
