use crate::network::http::client::{
    RequestFlags, RequestMetadata, add_unix_socket_flag, check_response_redirection,
    expand_unix_socket_path, http_client, http_client_pool, http_parse_redirect_mode,
    http_parse_url, request_add_authorization_header, request_add_custom_headers,
    request_handle_response, request_set_timeout, send_request_no_body,
};
use nu_engine::command_prelude::*;

use super::client::RedirectMode;

#[derive(Clone)]
pub struct HttpGet;

impl Command for HttpGet {
    fn name(&self) -> &str {
        "http get"
    }

    fn signature(&self) -> Signature {
        let sig = Signature::build("http get")
            .input_output_types(vec![(Type::Nothing, Type::Any)])
            .allow_variants_without_examples(true)
            .required(
                "URL",
                SyntaxShape::String,
                "The URL to fetch the contents from.",
            )
            .named(
                "user",
                SyntaxShape::Any,
                "The username when authenticating.",
                Some('u'),
            )
            .named(
                "password",
                SyntaxShape::Any,
                "The password when authenticating.",
                Some('p'),
            )
            .named(
                "max-time",
                SyntaxShape::Duration,
                "Max duration before timeout occurs.",
                Some('m'),
            )
            .named(
                "headers",
                SyntaxShape::Any,
                "Custom headers you want to add.",
                Some('H'),
            )
            .switch(
                "raw",
                "Fetch contents as text rather than a table.",
                Some('r'),
            )
            .switch(
                "insecure",
                "Allow insecure server connections when using SSL.",
                Some('k'),
            )
            .switch(
                "full",
                "Returns the full response instead of only the body.",
                Some('f'),
            )
            .switch(
                "allow-errors",
                "Do not fail if the server returns an error code.",
                Some('e'),
            )
            .switch("pool", "Using a global pool as a client.", None)
            .param(
                Flag::new("redirect-mode")
                    .short('R')
                    .arg(SyntaxShape::String)
                    .desc(
                        "What to do when encountering redirects. Default: 'follow'. Valid \
                         options: 'follow' ('f'), 'manual' ('m'), 'error' ('e').",
                    )
                    .completion(Completion::new_list(RedirectMode::MODES)),
            )
            .filter()
            .category(Category::Network);

        add_unix_socket_flag(sig)
    }

    fn description(&self) -> &str {
        "Fetch the contents from a URL."
    }

    fn extra_description(&self) -> &str {
        "Performs HTTP GET operation."
    }

    fn search_terms(&self) -> Vec<&str> {
        vec![
            "network", "fetch", "pull", "request", "download", "curl", "wget",
        ]
    }

    fn run(
        &self,
        engine_state: &EngineState,
        stack: &mut Stack,
        call: &Call,
        input: PipelineData,
    ) -> Result<PipelineData, ShellError> {
        run_get(engine_state, stack, call, input)
    }

    fn examples(&self) -> Vec<Example<'_>> {
        vec![
            Example {
                description: "Get content from example.com.",
                example: "http get https://www.example.com",
                result: None,
            },
            Example {
                description: "Get content from example.com, with username and password.",
                example: "http get --user myuser --password mypass https://www.example.com",
                result: None,
            },
            Example {
                description: "Get content from example.com, with custom header using a record.",
                example: "http get --headers {my-header-key: my-header-value} https://www.example.com",
                result: None,
            },
            Example {
                description: "Get content from example.com, with custom headers using a list.",
                example: "http get --headers [my-header-key-A my-header-value-A my-header-key-B my-header-value-B] https://www.example.com",
                result: None,
            },
            Example {
                description: "Get the response status code.",
                example: r#"http get https://www.example.com | metadata | get http_response.status"#,
                result: None,
            },
            Example {
                description: "Check response status while streaming.",
                example: r#"http get --allow-errors https://example.com/file | metadata access {|m| if $m.http_response.status != 200 { error make {msg: "failed"} } else { } } | lines"#,
                result: None,
            },
            Example {
                description: "Get from Docker daemon via Unix socket.",
                example: "http get --unix-socket /var/run/docker.sock http://localhost/containers/json",
                result: None,
            },
        ]
    }
}

struct Arguments {
    url: Value,
    headers: Option<Value>,
    raw: bool,
    insecure: bool,
    user: Option<String>,
    password: Option<String>,
    timeout: Option<Value>,
    full: bool,
    allow_errors: bool,
    redirect: Option<Spanned<String>>,
    unix_socket: Option<Spanned<String>>,
    pool: bool,
}

pub fn run_get(
    engine_state: &EngineState,
    stack: &mut Stack,
    call: &Call,
    _input: PipelineData,
) -> Result<PipelineData, ShellError> {
    let args = Arguments {
        url: call.req(engine_state, stack, 0)?,
        headers: call.get_flag(engine_state, stack, "headers")?,
        raw: call.has_flag(engine_state, stack, "raw")?,
        insecure: call.has_flag(engine_state, stack, "insecure")?,
        user: call.get_flag(engine_state, stack, "user")?,
        password: call.get_flag(engine_state, stack, "password")?,
        timeout: call.get_flag(engine_state, stack, "max-time")?,
        full: call.has_flag(engine_state, stack, "full")?,
        allow_errors: call.has_flag(engine_state, stack, "allow-errors")?,
        redirect: call.get_flag(engine_state, stack, "redirect-mode")?,
        unix_socket: call.get_flag(engine_state, stack, "unix-socket")?,
        pool: call.has_flag(engine_state, stack, "pool")?,
    };
    helper(engine_state, stack, call, args)
}

// Helper function that actually goes to retrieve the resource from the url given
// The Option<String> return a possible file extension which can be used in AutoConvert commands
fn helper(
    engine_state: &EngineState,
    stack: &mut Stack,
    call: &Call,
    args: Arguments,
) -> Result<PipelineData, ShellError> {
    let span = args.url.span();
    let Spanned {
        item: (requested_url, _),
        span: request_span,
    } = http_parse_url(call, span, args.url)?;
    let redirect_mode = http_parse_redirect_mode(args.redirect)?;

    let cwd = engine_state.cwd(None)?;
    let unix_socket_path = expand_unix_socket_path(args.unix_socket, &cwd);

    let mut request = if args.pool {
        http_client_pool(engine_state, stack)?.get(&requested_url)
    } else {
        let client = http_client(
            args.insecure,
            redirect_mode,
            unix_socket_path,
            engine_state,
            stack,
        )?;
        client.get(&requested_url)
    };

    request = request_set_timeout(args.timeout, request)?;
    request = request_add_authorization_header(args.user, args.password, request);
    request = request_add_custom_headers(args.headers, request)?;
    let (response, request_headers) =
        send_request_no_body(request, request_span, call.head, engine_state.signals());

    let request_flags = RequestFlags {
        raw: args.raw,
        full: args.full,
        allow_errors: args.allow_errors,
    };

    let response = response?;

    check_response_redirection(redirect_mode, span, &response)?;
    request_handle_response(
        engine_state,
        stack,
        RequestMetadata {
            requested_url: &requested_url,
            span,
            headers: request_headers,
            redirect_mode,
            flags: request_flags,
        },
        response,
    )
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_examples() {
        use crate::test_examples;

        test_examples(HttpGet {})
    }
}
