use super::util::{extend_record_with_metadata, parse_metadata_from_record};
use nu_engine::{ClosureEvalOnce, command_prelude::*};
use nu_protocol::{DataSource, DeprecationEntry, DeprecationType, ReportMode, engine::Closure};

#[derive(Clone)]
pub struct MetadataSet;

impl Command for MetadataSet {
    fn name(&self) -> &str {
        "metadata set"
    }

    fn description(&self) -> &str {
        "Set the metadata for items in the stream."
    }

    fn signature(&self) -> nu_protocol::Signature {
        Signature::build("metadata set")
            .input_output_types(vec![(Type::Any, Type::Any)])
            .optional(
                "closure",
                SyntaxShape::Closure(Some(vec![SyntaxShape::Record(vec![])])),
                "A closure that receives the current metadata and returns a new metadata record. Cannot be used with other flags.",
            )
            .switch(
                "datasource-ls",
                "(DEPRECATED) Assign the DataSource::Ls metadata to the input.",
                Some('l'),
            )
            .named(
                "datasource-filepath",
                SyntaxShape::Filepath,
                "Assign the DataSource::FilePath metadata to the input.",
                Some('f'),
            )
            .named(
                "path-columns",
                SyntaxShape::List(Box::new(SyntaxShape::String)),
                "A list of columns in the input for which path metadata will be assigned.",
                Some('p'),
            )
            .named(
                "content-type",
                SyntaxShape::String,
                "Assign content type metadata to the input.",
                Some('c'),
            )
            .named(
                "merge",
                SyntaxShape::Record(vec![]),
                "(DEPRECATED) Merge arbitrary metadata fields.",
                Some('m'),
            )
            .allow_variants_without_examples(true)
            .category(Category::Debug)
    }

    fn deprecation_info(&self) -> Vec<DeprecationEntry> {
        vec![
            DeprecationEntry {
                ty: DeprecationType::Flag("merge".into()),
                report_mode: ReportMode::FirstUse,
                since: Some("0.111.0".into()),
                expected_removal: Some("0.112.0".into()),
                help: Some(
                    "Use the closure parameter instead: `metadata set {|| merge {key: value}}`"
                        .into(),
                ),
            },
            DeprecationEntry {
                ty: DeprecationType::Flag("datasource-ls".into()),
                report_mode: ReportMode::FirstUse,
                since: Some("0.111.0".into()),
                expected_removal: Some("0.113.0".into()),
                help: Some(
                    "Use the path-columns flag instead: `metadata set --path-columns [name]`"
                        .into(),
                ),
            },
        ]
    }

    fn run(
        &self,
        engine_state: &EngineState,
        stack: &mut Stack,
        call: &Call,
        mut input: PipelineData,
    ) -> Result<PipelineData, ShellError> {
        let head = call.head;
        let closure: Option<Closure> = call.opt(engine_state, stack, 0)?;
        let ds_fp: Option<String> = call.get_flag(engine_state, stack, "datasource-filepath")?;
        let ds_ls = call.has_flag(engine_state, stack, "datasource-ls")?;
        let path_columns: Option<Vec<String>> =
            call.get_flag(engine_state, stack, "path-columns")?;
        let content_type: Option<String> = call.get_flag(engine_state, stack, "content-type")?;
        let merge: Option<Value> = call.get_flag(engine_state, stack, "merge")?;

        let mut metadata = match &mut input {
            PipelineData::Value(_, metadata)
            | PipelineData::ListStream(_, metadata)
            | PipelineData::ByteStream(_, metadata) => metadata.take().unwrap_or_default(),
            PipelineData::Empty => return Err(ShellError::PipelineEmpty { dst_span: head }),
        };

        // Handle closure parameter - mutually exclusive with flags
        if let Some(closure) = closure {
            if ds_fp.is_some()
                || ds_ls
                || path_columns.is_some()
                || content_type.is_some()
                || merge.is_some()
            {
                return Err(ShellError::GenericError {
                    error: "Incompatible parameters".into(),
                    msg: "cannot use closure with other flags".into(),
                    span: Some(head),
                    help: Some("Use either the closure parameter or flags, not both".into()),
                    inner: vec![],
                });
            }

            let record = extend_record_with_metadata(Record::new(), Some(&metadata), head);
            let metadata_value = record.into_value(head);

            let result = ClosureEvalOnce::new(engine_state, stack, closure)
                .run_with_value(metadata_value)?
                .into_value(head)?;

            let result_record = result.as_record().map_err(|err| ShellError::GenericError {
                error: "Closure must return a record".into(),
                msg: format!("got {}", result.get_type()),
                span: Some(head),
                help: Some("The closure should return a record with metadata fields".into()),
                inner: vec![err],
            })?;

            metadata = parse_metadata_from_record(result_record);
            return Ok(input.set_metadata(Some(metadata)));
        }

        if let Some(path_columns) = path_columns {
            metadata.path_columns = path_columns;
        }

        // Flag-based metadata modification
        if let Some(content_type) = content_type {
            metadata.content_type = Some(content_type);
        }

        if let Some(merge) = merge {
            let custom_record = merge.as_record()?;
            for (key, value) in custom_record {
                metadata.custom.insert(key.clone(), value.clone());
            }
        }

        match (ds_fp, ds_ls) {
            (Some(path), false) => metadata.data_source = DataSource::FilePath(path.into()),
            #[allow(deprecated)]
            (None, true) => {
                metadata.data_source = DataSource::Ls;
                metadata.path_columns.push("name".to_string());
            }
            (Some(_), true) => {
                return Err(ShellError::IncompatibleParameters {
                    left_message: "cannot use `--datasource-filepath`".into(),
                    left_span: call
                        .get_flag_span(stack, "datasource-filepath")
                        .expect("has flag"),
                    right_message: "with `--datasource-ls`".into(),
                    right_span: call
                        .get_flag_span(stack, "datasource-ls")
                        .expect("has flag"),
                });
            }
            (None, false) => (),
        }

        Ok(input.set_metadata(Some(metadata)))
    }

    fn examples(&self) -> Vec<Example<'_>> {
        vec![
            Example {
                description: "Set the metadata of a table literal so the `name` column is treated as a path.",
                example: "[[name color]; [Cargo.lock '#ff0000'] [Cargo.toml '#00ff00'] [README.md '#0000ff']] | metadata set --path-columns [name]",
                result: None,
            },
            Example {
                description: "Set the metadata of a file path.",
                example: "'crates' | metadata set --datasource-filepath $'(pwd)/crates'",
                result: None,
            },
            Example {
                description: "Set the content type metadata.",
                example: "'crates' | metadata set --content-type text/plain | metadata | get content_type",
                result: Some(Value::test_string("text/plain")),
            },
            Example {
                description: "Merge custom metadata.",
                example: r#""data" | metadata set {|| merge {custom_key: "value"}} | metadata | get custom_key"#,
                result: None,
            },
            Example {
                description: "Set metadata using a closure.",
                example: r#""data" | metadata set --content-type "text/csv" | metadata set {|m| $m | update content_type {$in + "-processed"}} | metadata | get content_type"#,
                result: Some(Value::test_string("text/csv-processed")),
            },
        ]
    }
}

#[cfg(test)]
mod test {
    use crate::{Metadata, test_examples_with_commands};

    use super::*;

    #[test]
    fn test_examples() {
        test_examples_with_commands(MetadataSet {}, &[&Metadata {}])
    }
}
