use crate::path::{Path, PathBuf, PathExt};
use std::collections::{BTreeSet, HashMap, HashSet};
use std::fmt::Display;
use std::fs;
use std::fs::File;
use std::io::Write;
#[cfg(unix)]
use std::os::unix::fs::symlink;
#[cfg(unix)]
use std::os::unix::prelude::*;
use std::sync::Mutex;
use std::time::Duration;

use bzip2::read::BzDecoder;
use color_eyre::eyre::{Context, Result};
use eyre::bail;
use filetime::{FileTime, set_file_times};
use flate2::read::GzDecoder;
use itertools::Itertools;
use std::sync::LazyLock as Lazy;
use tar::Archive;
use walkdir::WalkDir;
use zip::ZipArchive;

#[cfg(windows)]
use crate::config::Settings;
use crate::ui::progress_report::SingleReport;
use crate::{dirs, env};

pub fn open<P: AsRef<Path>>(path: P) -> Result<File> {
    let path = path.as_ref();
    trace!("open {}", display_path(path));
    File::open(path).wrap_err_with(|| format!("failed open: {}", display_path(path)))
}

pub fn read<P: AsRef<Path>>(path: P) -> Result<Vec<u8>> {
    let path = path.as_ref();
    trace!("cat {}", display_path(path));
    fs::read(path).wrap_err_with(|| format!("failed read: {}", display_path(path)))
}

pub fn size<P: AsRef<Path>>(path: P) -> Result<u64> {
    let path = path.as_ref();
    trace!("du -b {}", display_path(path));
    path.metadata()
        .map(|m| m.len())
        .wrap_err_with(|| format!("failed size: {}", display_path(path)))
}

pub fn append<P: AsRef<Path>, C: AsRef<[u8]>>(path: P, contents: C) -> Result<()> {
    let path = path.as_ref();
    trace!("append {}", display_path(path));
    fs::OpenOptions::new()
        .append(true)
        .create(true)
        .open(path)
        .and_then(|mut f| f.write_all(contents.as_ref()))
        .wrap_err_with(|| format!("failed append: {}", display_path(path)))
}

pub fn remove_all<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    match path.metadata().map(|m| m.file_type()) {
        Ok(x) if x.is_symlink() || x.is_file() => {
            remove_file(path)?;
        }
        Ok(x) if x.is_dir() => {
            trace!("rm -rf {}", display_path(path));
            fs::remove_dir_all(path)
                .wrap_err_with(|| format!("failed rm -rf: {}", display_path(path)))?;
        }
        _ => {}
    };
    Ok(())
}

pub fn remove_file_or_dir<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    match path.metadata().map(|m| m.file_type()) {
        Ok(x) if x.is_dir() => {
            remove_dir(path)?;
        }
        _ => {
            remove_file(path)?;
        }
    };
    Ok(())
}

pub fn remove_file<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    trace!("rm {}", display_path(path));
    fs::remove_file(path).wrap_err_with(|| format!("failed rm: {}", display_path(path)))
}

pub async fn remove_file_async<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    trace!("rm {}", display_path(path));
    tokio::fs::remove_file(path)
        .await
        .wrap_err_with(|| format!("failed rm: {}", display_path(path)))
}

pub fn remove_dir<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    (|| -> Result<()> {
        if path.exists() && is_empty_dir(path)? {
            trace!("rmdir {}", display_path(path));
            fs::remove_dir(path)?;
        }
        Ok(())
    })()
    .wrap_err_with(|| format!("failed to remove_dir: {}", display_path(path)))
}

pub fn remove_dir_ignore<P: AsRef<Path>>(path: P, is_empty_ignore_files: Vec<&str>) -> Result<()> {
    let path = path.as_ref();
    (|| -> Result<()> {
        if path.exists() && is_empty_dir_ignore(path, is_empty_ignore_files)? {
            trace!("rm -rf {}", display_path(path));
            remove_all_with_warning(path)?;
        }
        Ok(())
    })()
    .wrap_err_with(|| format!("failed to remove_dir: {}", display_path(path)))
}

pub fn remove_all_with_warning<P: AsRef<Path>>(path: P) -> Result<()> {
    remove_all(&path).map_err(|e| {
        warn!("failed to remove {}: {}", path.as_ref().display(), e);
        e
    })
}

pub fn rename<P: AsRef<Path>, Q: AsRef<Path>>(from: P, to: Q) -> Result<()> {
    let from = from.as_ref();
    let to = to.as_ref();
    trace!("mv {} {}", from.display(), to.display());
    fs::rename(from, to).wrap_err_with(|| {
        format!(
            "failed rename: {} -> {}",
            display_path(from),
            display_path(to)
        )
    })
}

pub fn copy<P: AsRef<Path>, Q: AsRef<Path>>(from: P, to: Q) -> Result<()> {
    let from = from.as_ref();
    let to = to.as_ref();
    trace!("cp {} {}", from.display(), to.display());
    fs::copy(from, to)
        .wrap_err_with(|| {
            format!(
                "failed copy: {} -> {}",
                display_path(from),
                display_path(to)
            )
        })
        .map(|_| ())
}

pub fn copy_dir_all<P: AsRef<Path>, Q: AsRef<Path>>(from: P, to: Q) -> Result<()> {
    let from = from.as_ref();
    let to = to.as_ref();
    trace!("cp -r {} {}", from.display(), to.display());
    recursive_ls(from)?.into_iter().try_for_each(|path| {
        let relative = path.strip_prefix(from)?;
        let dest = to.join(relative);
        create_dir_all(dest.parent().unwrap())?;
        copy(&path, &dest)?;
        Ok(())
    })
}

pub fn write<P: AsRef<Path>, C: AsRef<[u8]>>(path: P, contents: C) -> Result<()> {
    let path = path.as_ref();
    trace!("write {}", display_path(path));
    fs::write(path, contents).wrap_err_with(|| format!("failed write: {}", display_path(path)))
}
pub async fn write_async<P: AsRef<Path>, C: AsRef<[u8]>>(path: P, contents: C) -> Result<()> {
    let path = path.as_ref();
    trace!("write {}", display_path(path));
    tokio::fs::write(path, contents)
        .await
        .wrap_err_with(|| format!("failed write: {}", display_path(path)))
}

pub fn read_to_string<P: AsRef<Path>>(path: P) -> Result<String> {
    let path = path.as_ref();
    trace!("cat {}", path.display_user());
    fs::read_to_string(path)
        .wrap_err_with(|| format!("failed read_to_string: {}", path.display_user()))
}

pub async fn read_to_string_async<P: AsRef<Path>>(path: P) -> Result<String> {
    let path = path.as_ref();
    trace!("cat {}", path.display_user());
    tokio::fs::read_to_string(path)
        .await
        .wrap_err_with(|| format!("failed read_to_string: {}", path.display_user()))
}

pub fn create(path: &Path) -> Result<File> {
    if let Some(parent) = path.parent() {
        create_dir_all(parent)?;
    }
    trace!("touch {}", display_path(path));
    File::create(path).wrap_err_with(|| format!("failed create: {}", display_path(path)))
}

pub fn create_dir_all<P: AsRef<Path>>(path: P) -> Result<()> {
    static LOCK: Lazy<Mutex<u8>> = Lazy::new(Default::default);
    let _lock = LOCK.lock().unwrap();

    let path = path.as_ref();
    if !path.exists() {
        trace!("mkdir -p {}", display_path(path));
        if let Err(err) = fs::create_dir_all(path) {
            // if not exists error
            if err.kind() != std::io::ErrorKind::AlreadyExists {
                return Err(err)
                    .wrap_err_with(|| format!("failed create_dir_all: {}", display_path(path)));
            }
        }
    }
    Ok(())
}

/// replaces $HOME with "~"
pub fn display_path<P: AsRef<Path>>(path: P) -> String {
    path.as_ref().display_user()
}

pub fn display_rel_path<P: AsRef<Path>>(path: P) -> String {
    let path = path.as_ref();
    match path.strip_prefix(dirs::CWD.as_ref().unwrap()) {
        Ok(rel) => format!("./{}", rel.display()),
        Err(_) => display_path(path),
    }
}

/// replaces $HOME in a string with "~" and $PATH with "$PATH", generally used to clean up output
/// after it is rendered
pub fn replace_paths_in_string<S: Display>(input: S) -> String {
    let home = env::HOME.to_string_lossy().to_string();
    input.to_string().replace(&home, "~")
}

/// replaces "~" with $HOME
pub fn replace_path<P: AsRef<Path>>(path: P) -> PathBuf {
    let path = path.as_ref();
    match path.starts_with("~/") {
        true => dirs::HOME.join(path.strip_prefix("~/").unwrap()),
        false => path.to_path_buf(),
    }
}

pub fn touch_file(file: &Path) -> Result<()> {
    if !file.exists() {
        create(file)?;
        return Ok(());
    }
    trace!("touch_file {}", file.display());
    let now = FileTime::now();
    set_file_times(file, now, now)
        .wrap_err_with(|| format!("failed to touch file: {}", display_path(file)))
}

pub fn touch_dir(dir: &Path) -> Result<()> {
    trace!("touch {}", dir.display());
    let now = FileTime::now();
    set_file_times(dir, now, now)
        .wrap_err_with(|| format!("failed to touch dir: {}", display_path(dir)))
}

/// Synchronizes a directory to disk, ensuring that filesystem metadata changes
/// (such as file creations or deletions) are persisted.
///
/// This is important after operations like removing files to ensure the changes
/// are immediately visible to other processes, e.g. to avoid race conditions.
///
/// # Platform-specific behavior
///
/// - **Unix/Linux**: Performs an fsync on the directory file descriptor, which
///   ensures directory metadata (like file listings) is written to disk.
/// - **Windows**: Not implemented (no-op).
///
/// # Errors
///
/// On Unix systems, returns an error if the directory cannot be opened or synced.
/// On Windows, always succeeds.
#[cfg(unix)]
pub fn sync_dir<P: AsRef<Path>>(path: P) -> Result<()> {
    let path = path.as_ref();
    trace!("sync {}", display_path(path));
    let dir = File::open(path)
        .wrap_err_with(|| format!("failed to open dir for sync: {}", display_path(path)))?;
    dir.sync_all()
        .wrap_err_with(|| format!("failed to sync dir: {}", display_path(path)))
}

#[cfg(windows)]
pub fn sync_dir<P: AsRef<Path>>(_path: P) -> Result<()> {
    // Not implemented on Windows
    Ok(())
}

pub fn modified_duration(path: &Path) -> Result<Duration> {
    let metadata = path.metadata()?;
    let modified = metadata.modified()?;
    let duration = modified.elapsed().unwrap_or_default();
    Ok(duration)
}

pub fn find_up<FN: AsRef<str>>(from: &Path, filenames: &[FN]) -> Option<PathBuf> {
    let mut current = from.to_path_buf();
    loop {
        for filename in filenames {
            let path = current.join(filename.as_ref());
            if path.exists() {
                return Some(path);
            }
        }
        if !current.pop() {
            return None;
        }
    }
}

pub fn dir_subdirs(dir: &Path) -> Result<BTreeSet<String>> {
    let mut output = Default::default();

    if !dir.exists() {
        return Ok(output);
    }

    for entry in dir.read_dir()? {
        let entry = entry?;
        let ft = entry.file_type()?;
        if ft.is_dir() || (ft.is_symlink() && entry.path().is_dir()) {
            output.insert(entry.file_name().into_string().unwrap());
        }
    }

    Ok(output)
}

pub fn ls(dir: &Path) -> Result<BTreeSet<PathBuf>> {
    let mut output = Default::default();

    if !dir.is_dir() {
        return Ok(output);
    }

    for entry in dir.read_dir()? {
        let entry = entry?;
        output.insert(entry.path());
    }

    Ok(output)
}

pub fn recursive_ls(dir: &Path) -> Result<BTreeSet<PathBuf>> {
    if !dir.is_dir() {
        return Ok(Default::default());
    }

    Ok(WalkDir::new(dir)
        .follow_links(true)
        .into_iter()
        .filter_ok(|e| e.file_type().is_file())
        .map_ok(|e| e.path().to_path_buf())
        .try_collect()?)
}

#[cfg(unix)]
pub fn make_symlink(target: &Path, link: &Path) -> Result<(PathBuf, PathBuf)> {
    trace!("ln -sf {} {}", target.display(), link.display());
    if link.is_file() || link.is_symlink() {
        fs::remove_file(link)?;
    }
    symlink(target, link)
        .wrap_err_with(|| format!("failed to ln -sf {} {}", target.display(), link.display()))?;
    Ok((target.to_path_buf(), link.to_path_buf()))
}

#[cfg(unix)]
pub fn make_symlink_or_copy(target: &Path, link: &Path) -> Result<()> {
    make_symlink(target, link)?;
    Ok(())
}

#[cfg(windows)]
pub fn make_symlink_or_copy(target: &Path, link: &Path) -> Result<()> {
    copy(target, link)?;
    Ok(())
}

#[cfg(windows)]
pub fn make_symlink(target: &Path, link: &Path) -> Result<(PathBuf, PathBuf)> {
    if let Err(err) = junction::create(target, link) {
        if err.kind() == std::io::ErrorKind::AlreadyExists {
            let _ = fs::remove_file(link);
            junction::create(target, link)
        } else {
            Err(err)
        }
    } else {
        Ok(())
    }
    .wrap_err_with(|| format!("failed to ln -sf {} {}", target.display(), link.display()))?;
    Ok((target.to_path_buf(), link.to_path_buf()))
}

#[cfg(windows)]
pub fn make_symlink_or_file(target: &Path, link: &Path) -> Result<()> {
    trace!("ln -sf {} {}", target.display(), link.display());
    if link.is_file() || link.is_symlink() {
        // remove existing file if exists
        fs::remove_file(link)?;
    }
    xx::file::write(link, target.to_string_lossy().to_string())?;
    Ok(())
}

pub fn resolve_symlink(link: &Path) -> Result<Option<PathBuf>> {
    // Windows symlink are write in file currently
    // may be changed to symlink in the future
    if link.is_symlink() {
        Ok(Some(fs::read_link(link)?))
    } else if link.is_file() {
        Ok(Some(fs::read_to_string(link)?.into()))
    } else {
        Ok(None)
    }
}

#[cfg(unix)]
pub fn make_symlink_or_file(target: &Path, link: &Path) -> Result<()> {
    make_symlink(target, link)?;
    Ok(())
}

pub fn remove_symlinks_with_target_prefix(
    symlink_dir: &Path,
    target_prefix: &Path,
) -> Result<Vec<PathBuf>> {
    if !symlink_dir.exists() {
        return Ok(vec![]);
    }
    let mut removed = vec![];
    for entry in symlink_dir.read_dir()? {
        let entry = entry?;
        let path = entry.path();
        if path.is_symlink() {
            let target = path.read_link()?;
            if target.starts_with(target_prefix) {
                fs::remove_file(&path)?;
                removed.push(path);
            }
        }
    }
    Ok(removed)
}

#[cfg(unix)]
pub fn is_executable(path: &Path) -> bool {
    if let Ok(metadata) = path.metadata() {
        return metadata.permissions().mode() & 0o111 != 0;
    }
    false
}

#[cfg(windows)]
pub fn is_executable(path: &Path) -> bool {
    if has_known_executable_extension(path) {
        return true;
    }
    has_shebang(path)
}

#[cfg(windows)]
pub fn has_known_executable_extension(path: &Path) -> bool {
    path.extension().map_or(
        Settings::get()
            .windows_executable_extensions
            .contains(&String::new()),
        |ext| {
            if let Some(str_val) = ext.to_str() {
                return Settings::get()
                    .windows_executable_extensions
                    .contains(&str_val.to_lowercase().to_string());
            }
            false
        },
    )
}

/// Check if a file starts with a shebang (#!).
/// Only reads the first 2 bytes to minimize I/O during task discovery.
#[cfg(windows)]
pub fn has_shebang(path: &Path) -> bool {
    std::fs::File::open(path)
        .and_then(|mut f| {
            use std::io::Read;
            let mut buf = [0u8; 2];
            f.read_exact(&mut buf)?;
            Ok(buf == *b"#!")
        })
        .unwrap_or(false)
}

#[cfg(unix)]
pub fn make_executable<P: AsRef<Path>>(path: P) -> Result<()> {
    trace!("chmod +x {}", display_path(&path));
    let path = path.as_ref();
    let mut perms = path.metadata()?.permissions();
    perms.set_mode(perms.mode() | 0o111);
    fs::set_permissions(path, perms)
        .wrap_err_with(|| format!("failed to chmod +x: {}", display_path(path)))?;
    Ok(())
}

#[cfg(windows)]
pub fn make_executable<P: AsRef<Path>>(_path: P) -> Result<()> {
    Ok(())
}

#[cfg(unix)]
pub async fn make_executable_async<P: AsRef<Path>>(path: P) -> Result<()> {
    trace!("chmod +x {}", display_path(&path));
    let path = path.as_ref();
    let mut perms = path.metadata()?.permissions();
    perms.set_mode(perms.mode() | 0o111);
    tokio::fs::set_permissions(path, perms)
        .await
        .wrap_err_with(|| format!("failed to chmod +x: {}", display_path(path)))
}

#[cfg(windows)]
pub async fn make_executable_async<P: AsRef<Path>>(_path: P) -> Result<()> {
    Ok(())
}

pub fn all_dirs<P: AsRef<Path>>(
    start_dir: P,
    ceiling_dirs: &HashSet<PathBuf>,
) -> Result<Vec<PathBuf>> {
    trace!(
        "file::all_dirs Collecting all ancestors of {} until ceiling {:?}",
        display_path(&start_dir),
        ceiling_dirs
    );
    Ok(start_dir
        .as_ref()
        .ancestors()
        .map_while(|p| {
            if ceiling_dirs.contains(p) {
                debug!(
                    "file::all_dirs Reached ceiling directory: {}",
                    display_path(p)
                );
                None
            } else {
                trace!(
                    "file::all_dirs Adding ancestor directory: {}",
                    display_path(p)
                );
                Some(p.to_path_buf())
            }
        })
        .collect())
}

fn is_empty_dir(path: &Path) -> Result<bool> {
    path.read_dir()
        .map(|mut i| i.next().is_none())
        .wrap_err_with(|| format!("failed to read_dir: {}", display_path(path)))
}

fn is_empty_dir_ignore(path: &Path, ignore_files: Vec<&str>) -> Result<bool> {
    path.read_dir()
        .map(|mut i| {
            i.all(|entry| match entry {
                Ok(entry) => ignore_files.iter().any(|ignore_file| {
                    entry
                        .file_name()
                        .to_string_lossy()
                        .eq_ignore_ascii_case(ignore_file)
                }),
                Err(_) => false,
            })
        })
        .wrap_err_with(|| format!("failed to read_dir: {}", display_path(path)))
}

pub struct FindUp {
    current_dir: PathBuf,
    current_dir_filenames: Vec<String>,
    filenames: Vec<String>,
}

impl FindUp {
    pub fn new(from: &Path, filenames: &[String]) -> Self {
        let filenames: Vec<String> = filenames.iter().map(|s| s.to_string()).collect();
        Self {
            current_dir: from.to_path_buf(),
            filenames: filenames.clone(),
            current_dir_filenames: filenames,
        }
    }
}

impl Iterator for FindUp {
    type Item = PathBuf;

    fn next(&mut self) -> Option<Self::Item> {
        while let Some(filename) = self.current_dir_filenames.pop() {
            let path = self.current_dir.join(filename);
            if path.is_file() {
                return Some(path);
            }
        }
        self.current_dir_filenames.clone_from(&self.filenames);
        if cfg!(test) && self.current_dir == *dirs::HOME {
            return None; // in tests, do not recurse further than ./test
        }
        if !self.current_dir.pop() {
            return None;
        }
        self.next()
    }
}

/// returns the first executable in PATH
/// will not include mise bin paths or other paths added by mise
pub fn which<P: AsRef<Path>>(name: P) -> Option<PathBuf> {
    static CACHE: Lazy<Mutex<HashMap<PathBuf, Option<PathBuf>>>> = Lazy::new(Default::default);

    let name = name.as_ref();
    if let Some(path) = CACHE.lock().unwrap().get(name) {
        return path.clone();
    }
    let path = _which(name, &env::PATH);
    CACHE
        .lock()
        .unwrap()
        .insert(name.to_path_buf(), path.clone());
    path
}

/// returns the first executable in PATH
/// will include mise bin paths or other paths added by mise
pub fn which_non_pristine<P: AsRef<Path>>(name: P) -> Option<PathBuf> {
    _which(name, &env::PATH_NON_PRISTINE)
}

/// returns the first executable in PATH, excluding the mise shim directory
/// use this for internal tool lookups to avoid recursive shim invocations
/// (shims call `mise exec`, which would re-enter the same code path)
pub fn which_no_shims<P: AsRef<Path>>(name: P) -> Option<PathBuf> {
    let shim_dir = &*dirs::SHIMS;
    let paths: Vec<PathBuf> = env::PATH_NON_PRISTINE
        .iter()
        .filter(|p| p.as_path() != *shim_dir)
        .cloned()
        .collect();
    _which(name, &paths)
}

fn _which<P: AsRef<Path>>(name: P, paths: &[PathBuf]) -> Option<PathBuf> {
    let name = name.as_ref();
    paths.iter().find_map(|path| {
        let bin = path.join(name);
        if is_executable(&bin) { Some(bin) } else { None }
    })
}

pub fn un_gz(input: &Path, dest: &Path) -> Result<()> {
    debug!("gunzip {} > {}", input.display(), dest.display());
    let f = File::open(input)?;
    let mut dec = GzDecoder::new(f);
    let mut output = File::create(dest)?;
    std::io::copy(&mut dec, &mut output)
        .wrap_err_with(|| format!("failed to un-gzip: {}", display_path(input)))?;
    Ok(())
}

pub fn un_xz(input: &Path, dest: &Path) -> Result<()> {
    debug!("xz -d {} -c > {}", input.display(), dest.display());
    let f = File::open(input)?;
    let mut dec = xz2::read::XzDecoder::new(f);
    let mut output = File::create(dest)?;
    std::io::copy(&mut dec, &mut output)
        .wrap_err_with(|| format!("failed to un-xz: {}", display_path(input)))?;
    Ok(())
}

pub fn un_zst(input: &Path, dest: &Path) -> Result<()> {
    debug!("zstd -d {} -c > {}", input.display(), dest.display());
    let f = File::open(input)?;
    let mut dec = zstd::Decoder::new(f)?;
    let mut output = File::create(dest)?;
    std::io::copy(&mut dec, &mut output)
        .wrap_err_with(|| format!("failed to un-zst: {}", display_path(input)))?;
    Ok(())
}

pub fn un_bz2(input: &Path, dest: &Path) -> Result<()> {
    debug!("bzip2 -d {} -c > {}", input.display(), dest.display());
    let f = File::open(input)?;
    let mut dec = BzDecoder::new(f);
    let mut output = File::create(dest)?;
    std::io::copy(&mut dec, &mut output)
        .wrap_err_with(|| format!("failed to un-bz2: {}", display_path(input)))?;
    Ok(())
}

#[derive(Debug, Clone, Copy, PartialEq, strum::EnumString, strum::Display)]
pub enum TarFormat {
    #[strum(serialize = "tar.gz", serialize = "tgz")]
    TarGz,
    #[strum(serialize = "gz")]
    Gz,
    #[strum(serialize = "tar.xz", serialize = "txz")]
    TarXz,
    #[strum(serialize = "xz")]
    Xz,
    #[strum(serialize = "tar.bz2", serialize = "tbz2")]
    TarBz2,
    #[strum(serialize = "bz2")]
    Bz2,
    #[strum(serialize = "tar.zst", serialize = "tzst")]
    TarZst,
    #[strum(serialize = "zst")]
    Zst,
    #[strum(serialize = "tar")]
    Tar,
    #[strum(serialize = "zip", serialize = "vsix")]
    Zip,
    #[strum(serialize = "7z")]
    SevenZip,
    #[strum(serialize = "raw")]
    Raw,
}

impl TarFormat {
    pub fn from_file_name(filename: &str) -> Self {
        let filename = filename.to_lowercase();

        if let Some(idx) = filename.rfind(".tar.") {
            let ext = &filename[idx + 1..];
            let fmt = Self::from_ext(ext);
            if fmt != TarFormat::Raw {
                return fmt;
            }
        }

        if let Some(ext) = Path::new(&filename).extension().and_then(|s| s.to_str()) {
            Self::from_ext(ext)
        } else {
            TarFormat::Raw
        }
    }

    pub fn from_ext(ext: &str) -> Self {
        ext.to_lowercase().parse().unwrap_or(TarFormat::Raw)
    }

    pub fn is_archive(&self) -> bool {
        match self {
            TarFormat::TarGz
            | TarFormat::TarXz
            | TarFormat::TarBz2
            | TarFormat::TarZst
            | TarFormat::Tar
            | TarFormat::Zip
            | TarFormat::SevenZip => true,
            TarFormat::Gz | TarFormat::Xz | TarFormat::Bz2 | TarFormat::Zst | TarFormat::Raw => {
                false
            }
        }
    }

    pub fn extension(&self) -> Option<&'static str> {
        match self {
            TarFormat::TarGz => Some("tar.gz"),
            TarFormat::Gz => Some("gz"),
            TarFormat::TarXz => Some("tar.xz"),
            TarFormat::Xz => Some("xz"),
            TarFormat::TarBz2 => Some("tar.bz2"),
            TarFormat::Bz2 => Some("bz2"),
            TarFormat::TarZst => Some("tar.zst"),
            TarFormat::Zst => Some("zst"),
            TarFormat::Tar => Some("tar"),
            TarFormat::Zip => Some("zip"),
            TarFormat::SevenZip => Some("7z"),
            TarFormat::Raw => None,
        }
    }
}

pub struct TarOptions<'a> {
    pub format: TarFormat,
    pub strip_components: usize,
    pub pr: Option<&'a dyn SingleReport>,
    /// When false, files will be extracted with current timestamp instead of archive's mtime
    pub preserve_mtime: bool,
}

impl<'a> TarOptions<'a> {
    pub fn new(format: TarFormat) -> Self {
        Self {
            format,
            strip_components: 0,
            pr: None,
            preserve_mtime: true,
        }
    }
}

pub fn untar(archive: &Path, dest: &Path, opts: &TarOptions) -> Result<()> {
    if opts.format == TarFormat::Zip {
        return unzip(
            archive,
            dest,
            &ZipOptions {
                strip_components: opts.strip_components,
            },
        );
    } else if opts.format == TarFormat::SevenZip {
        #[cfg(windows)]
        return un7z(
            archive,
            dest,
            &SevenZipOptions {
                strip_components: opts.strip_components,
            },
        );
    }

    debug!("tar -xf {} -C {}", archive.display(), dest.display());
    if let Some(pr) = &opts.pr {
        pr.set_message(format!(
            "extract {}",
            archive.file_name().unwrap().to_string_lossy()
        ));
    }

    let err = || {
        let archive = display_path(archive);
        let dest = display_path(dest);
        format!("failed to extract tar: {archive} to {dest}")
    };

    let format = opts.format;
    if !format.is_archive() && format != TarFormat::Raw {
        let mut reader = open_tar(format, archive)?;
        // If dest is a directory, join with the archive filename (minus extension)
        // If dest is not a dir, assume it's the target file path
        let out_path = if dest.is_dir() {
            let name = archive
                .file_stem()
                .unwrap_or_else(|| archive.file_name().unwrap());
            dest.join(name)
        } else {
            dest.to_path_buf()
        };

        if let Some(parent) = out_path.parent() {
            create_dir_all(parent).wrap_err_with(err)?;
        }
        let mut out = File::create(&out_path).wrap_err_with(err)?;
        std::io::copy(&mut reader, &mut out).wrap_err_with(err)?;
        return Ok(());
    }

    let tar = open_tar(format, archive)?;
    // TODO: put this back in when we can read+write in parallel
    // let mut cur = Cursor::new(vec![]);
    // let mut total = 0;
    // loop {
    //     let mut buf = Cursor::new(vec![0; 1024 * 1024]);
    //     let n = tar.read(buf.get_mut()).wrap_err_with(err)?;
    //     cur.get_mut().extend_from_slice(&buf.get_ref()[..n]);
    //     if n == 0 {
    //         break;
    //     }
    //     if let Some(pr) = &opts.pr {
    //         total += n as u64;
    //         pr.set_length(total);
    //     }
    // }
    create_dir_all(dest).wrap_err_with(err)?;

    // Try to extract using the tar crate, detecting sparse files during extraction
    let mut needs_system_tar = false;
    for entry in Archive::new(tar).entries().wrap_err_with(err)? {
        let mut entry = entry.wrap_err_with(err)?;

        // Check if this is a GNU sparse file
        if entry.header().entry_type().is_gnu_sparse() {
            debug!("Detected GNU sparse file, falling back to system tar");
            needs_system_tar = true;
            // Clean up any partial extraction
            remove_all(dest)?;
            create_dir_all(dest)?;
            break;
        }

        // Configure mtime preservation based on options
        entry.set_preserve_mtime(opts.preserve_mtime);

        trace!("extracting {}", entry.path().wrap_err_with(err)?.display());
        entry.unpack_in(dest).wrap_err_with(err)?;
    }

    // Check for the GNUSparseFile.0 directory which indicates the tar crate
    // incorrectly handled a sparse file
    if !needs_system_tar {
        let sparse_dir = dest.join("GNUSparseFile.0");
        if sparse_dir.exists() && sparse_dir.is_dir() {
            debug!("Found GNUSparseFile.0 directory, using system tar");
            needs_system_tar = true;
            // Clean up the bad extraction
            remove_all(dest)?;
            create_dir_all(dest)?;
        }
    }

    if needs_system_tar {
        // Use system tar for archives with problematic sparse files
        // The tar crate doesn't properly handle certain GNU sparse formats
        debug!("Using system tar for: {}", archive.display());

        // When preserve_mtime is false, use -m flag to not restore modification times
        // This causes extracted files to have current time, which is important for
        // cache invalidation and autopruning. Works on both BSD and GNU tar.
        if !opts.preserve_mtime {
            cmd!("tar", "-mxf", archive, "-C", dest)
                .run()
                .wrap_err_with(|| {
                    format!("Failed to extract {} using system tar", archive.display())
                })?;
        } else {
            cmd!("tar", "-xf", archive, "-C", dest)
                .run()
                .wrap_err_with(|| {
                    format!("Failed to extract {} using system tar", archive.display())
                })?;
        }
    }

    // Always use our manual strip to ensure consistent behavior across backends
    strip_archive_path_components(dest, opts.strip_components).wrap_err_with(err)?;
    Ok(())
}

fn open_tar(format: TarFormat, archive: &Path) -> Result<Box<dyn std::io::Read>> {
    let f = File::open(archive)?;
    Ok(match format {
        // TODO: we probably shouldn't assume raw is tar.gz, but this was to retain existing behavior
        TarFormat::TarGz | TarFormat::Gz | TarFormat::Raw => Box::new(GzDecoder::new(f)),
        TarFormat::TarXz | TarFormat::Xz => Box::new(xz2::read::XzDecoder::new(f)),
        TarFormat::TarBz2 | TarFormat::Bz2 => Box::new(BzDecoder::new(f)),
        TarFormat::TarZst | TarFormat::Zst => Box::new(zstd::stream::read::Decoder::new(f)?),
        TarFormat::Tar => Box::new(f),
        TarFormat::Zip => bail!("zip format not supported"),
        TarFormat::SevenZip => bail!("7z format not supported"),
    })
}

fn strip_archive_path_components(dir: &Path, strip_depth: usize) -> Result<()> {
    if strip_depth == 0 {
        return Ok(());
    }
    if strip_depth > 1 {
        bail!("strip-components > 1 is not supported");
    }

    let top_level_paths = ls(dir)?;

    for path in top_level_paths {
        if !path.symlink_metadata()?.is_dir() {
            continue;
        }

        // rename the directory to a temp name to avoid conflicts when moving files
        let temp_path = path.with_file_name(format!(
            "{}_tmp_strip",
            path.file_name().unwrap().to_string_lossy()
        ));
        fs::rename(&path, &temp_path)?;

        for entry in ls(&temp_path)? {
            if let Some(file_name) = entry.file_name() {
                let dest_path = dir.join(file_name);
                fs::rename(entry, dest_path)?;
            } else {
                continue;
            }
        }

        remove_dir(temp_path)?;
    }
    Ok(())
}

#[derive(Default)]
pub struct ZipOptions {
    pub strip_components: usize,
}

pub fn unzip(archive: &Path, dest: &Path, opts: &ZipOptions) -> Result<()> {
    // TODO: show progress
    debug!("unzip {} -d {}", archive.display(), dest.display());
    ZipArchive::new(File::open(archive)?)
        .wrap_err_with(|| format!("failed to open zip archive: {}", display_path(archive)))?
        .extract(dest)
        .wrap_err_with(|| format!("failed to extract zip archive: {}", display_path(archive)))?;

    strip_archive_path_components(dest, opts.strip_components).wrap_err_with(|| {
        format!(
            "failed to strip path components from zip archive: {}",
            display_path(archive)
        )
    })
}

pub fn un_dmg(archive: &Path, dest: &Path) -> Result<()> {
    debug!(
        "hdiutil attach -quiet -nobrowse -mountpoint {} {}",
        dest.display(),
        archive.display()
    );
    let tmp = tempfile::TempDir::new()?;
    cmd!(
        "hdiutil",
        "attach",
        "-quiet",
        "-nobrowse",
        "-mountpoint",
        tmp.path(),
        archive.to_path_buf()
    )
    .run()?;
    copy_dir_all(tmp.path(), dest)?;
    cmd!("hdiutil", "detach", tmp.path()).run()?;
    Ok(())
}

pub fn un_pkg(archive: &Path, dest: &Path) -> Result<()> {
    debug!(
        "pkgutil --expand-full {} {}",
        archive.display(),
        dest.display()
    );
    cmd!("pkgutil", "--expand-full", archive, dest).run()?;
    Ok(())
}

#[cfg(windows)]
#[derive(Default)]
pub struct SevenZipOptions {
    pub strip_components: usize,
}

#[cfg(windows)]
pub fn un7z(archive: &Path, dest: &Path, opts: &SevenZipOptions) -> Result<()> {
    sevenz_rust::decompress_file(archive, dest)
        .wrap_err_with(|| format!("failed to extract 7z archive: {}", display_path(archive)))?;

    strip_archive_path_components(dest, opts.strip_components).wrap_err_with(|| {
        format!(
            "failed to strip path components from 7z archive: {}",
            display_path(archive)
        )
    })
}

pub fn split_file_name(path: &Path) -> (String, String) {
    let file_name = path.file_name().unwrap().to_string_lossy();
    let (file_name_base, ext) = file_name
        .split_once('.')
        .unwrap_or((file_name.as_ref(), ""));
    (file_name_base.to_string(), ext.to_string())
}

pub fn same_file(a: &Path, b: &Path) -> bool {
    desymlink_path(a) == desymlink_path(b)
}

pub fn desymlink_path(p: &Path) -> PathBuf {
    if p.is_symlink()
        && let Ok(target) = fs::read_link(p)
    {
        return target
            .canonicalize()
            .unwrap_or_else(|_| target.to_path_buf());
    }
    p.canonicalize().unwrap_or_else(|_| p.to_path_buf())
}

pub fn clone_dir(from: &PathBuf, to: &PathBuf) -> Result<()> {
    if cfg!(macos) {
        cmd!("cp", "-cR", from, to).run()?;
    } else if cfg!(windows) {
        cmd!("robocopy", from, to, "/MIR").run()?;
    } else {
        cmd!("cp", "--reflink=auto", "-r", from, to).run()?;
    }
    Ok(())
}

/// Inspects the top-level contents of a tar archive without extracting it
/// Skips leading CurDir (".") components from a path's components iterator.
/// Archives often have paths like "./foo/bar" where the leading "." should be ignored.
fn skip_curdir_components(path: &Path) -> impl Iterator<Item = std::path::Component<'_>> {
    path.components()
        .skip_while(|c| matches!(c, std::path::Component::CurDir))
}

pub fn inspect_tar_contents(archive: &Path, format: TarFormat) -> Result<Vec<(String, bool)>> {
    let tar = open_tar(format, archive)?;
    let mut archive = Archive::new(tar);
    let mut top_level_components = std::collections::HashMap::new();

    for entry in archive.entries()? {
        let entry = entry?;
        let path = entry.path()?;
        let header = entry.header();

        // Get the first non-CurDir component of the path (top-level directory/file)
        let mut components = skip_curdir_components(&path);

        if let Some(first_component) = components.next() {
            let name = first_component.as_os_str().to_string_lossy().to_string();

            // Check if this entry indicates the component is a directory
            // It's a directory if the entry type is dir OR if there are more components after the first
            let is_directory = header.entry_type().is_dir() || components.next().is_some();

            // Update the component's directory status
            // A component is a directory if ANY entry indicates it's a directory
            let existing = top_level_components.entry(name.clone()).or_insert(false);
            *existing = *existing || is_directory;
        }
    }

    Ok(top_level_components.into_iter().collect())
}

/// Inspects the top-level contents of a zip archive without extracting it
pub fn inspect_zip_contents(archive: &Path) -> Result<Vec<(String, bool)>> {
    let f = File::open(archive)?;
    let mut archive = ZipArchive::new(f)
        .wrap_err_with(|| format!("failed to open zip archive: {}", display_path(archive)))?;
    let mut top_level_components = std::collections::HashMap::new();

    for i in 0..archive.len() {
        let file = archive.by_index(i)?;
        if let Some(path) = file.enclosed_name() {
            // Get the first non-CurDir component of the path (top-level directory/file)
            let mut components = skip_curdir_components(&path);

            if let Some(first_component) = components.next() {
                let name = first_component.as_os_str().to_string_lossy().to_string();

                // Check if this entry indicates the component is a directory
                // It's a directory if the entry type is dir OR if there are more components after the first
                let is_directory = file.is_dir() || components.next().is_some();

                let existing = top_level_components.entry(name.clone()).or_insert(false);
                *existing = *existing || is_directory;
            }
        }
    }

    Ok(top_level_components.into_iter().collect())
}

/// Adapted from inspect_tar_contents for 7z archives
#[cfg(windows)]
pub fn inspect_7z_contents(archive: &Path) -> Result<Vec<(String, bool)>> {
    let sevenz = sevenz_rust::SevenZReader::open(archive, sevenz_rust::Password::empty())?;
    let mut top_level_components = std::collections::HashMap::new();

    for file in &sevenz.archive().files {
        let path = PathBuf::from(file.name());

        // Get the first non-CurDir component of the path (top-level directory/file)
        let mut components = skip_curdir_components(&path);

        if let Some(first_component) = components.next() {
            let name = first_component.as_os_str().to_string_lossy().to_string();
            // It's a directory if the entry type is dir OR if there are more components after the first
            let is_directory = file.is_directory() || components.next().is_some();

            let existing = top_level_components.entry(name.clone()).or_insert(false);
            *existing = *existing || is_directory;
        }
    }

    Ok(top_level_components.into_iter().collect())
}

#[cfg(not(windows))]
pub fn inspect_7z_contents(_archive: &Path) -> Result<Vec<(String, bool)>> {
    unimplemented!("7z format not supported on this platform")
}

/// Determines if strip_components=1 should be applied based on archive structure
pub fn should_strip_components(archive: &Path, format: TarFormat) -> Result<bool> {
    let top_level_entries = match format {
        TarFormat::Zip => inspect_zip_contents(archive)?,
        TarFormat::SevenZip => inspect_7z_contents(archive)?,
        _ => inspect_tar_contents(archive, format)?,
    };

    // If there's exactly one top-level entry and it's a directory, we should strip it
    if top_level_entries.len() == 1 {
        let (_, is_directory) = &top_level_entries[0];
        Ok(*is_directory)
    } else {
        Ok(false)
    }
}

#[cfg(test)]
mod tests {
    use pretty_assertions::assert_eq;

    use crate::config::Config;

    use super::*;

    #[tokio::test]
    async fn test_find_up() {
        let _config = Config::get().await.unwrap();
        let path = &env::current_dir().unwrap();
        let filenames = vec![".miserc", ".mise.toml", ".test-tool-versions"]
            .into_iter()
            .map(|s| s.to_string())
            .collect_vec();
        #[allow(clippy::needless_collect)]
        let find_up = FindUp::new(path, &filenames).collect::<Vec<_>>();
        let mut find_up = find_up.into_iter();
        assert_eq!(
            find_up.next(),
            Some(dirs::HOME.join("cwd/.test-tool-versions"))
        );
        assert_eq!(find_up.next(), Some(dirs::HOME.join(".test-tool-versions")));
    }

    #[tokio::test]
    async fn test_find_up_2() {
        let _config = Config::get().await.unwrap();
        let path = &dirs::HOME.join("fixtures");
        let filenames = vec![".test-tool-versions"];
        let result = find_up(path, &filenames);
        assert_eq!(result, Some(dirs::HOME.join(".test-tool-versions")));
    }

    #[tokio::test]
    async fn test_dir_subdirs() {
        let _config = Config::get().await.unwrap();
        let subdirs = dir_subdirs(&dirs::HOME).unwrap();
        assert!(subdirs.contains("cwd"));
    }

    #[tokio::test]
    #[cfg(unix)]
    async fn test_display_path() {
        let _config = Config::get().await.unwrap();
        use std::ops::Deref;
        let path = dirs::HOME.join("cwd");
        assert_eq!(display_path(path), "~/cwd");

        let path = Path::new("/tmp")
            .join(dirs::HOME.deref().strip_prefix("/").unwrap())
            .join("cwd");
        assert_eq!(display_path(&path), path.display().to_string());
    }

    #[tokio::test]
    async fn test_replace_path() {
        let _config = Config::get().await.unwrap();
        assert_eq!(replace_path(Path::new("~/cwd")), dirs::HOME.join("cwd"));
        assert_eq!(replace_path(Path::new("/cwd")), Path::new("/cwd"));
    }

    #[test]
    fn test_should_strip_components() {
        // Test that the function correctly identifies when to strip components
        // This is a basic test to ensure the logic works correctly

        // For now, we'll test with a non-existent file to ensure the function
        // returns false when it can't read the archive
        let non_existent_path = Path::new("/non/existent/archive.tar.gz");
        let result = should_strip_components(non_existent_path, TarFormat::TarGz);
        assert!(result.is_err()); // Should fail to open non-existent file

        // Note: To properly test this function, we would need actual tar archives
        // with different structures (single file, single directory, multiple entries)
        // This would require creating test fixtures, which is beyond the scope
        // of this fix. The important thing is that the logic now correctly
        // checks if the single entry is a directory before deciding to strip.
    }

    #[test]
    fn test_inspect_tar_contents_logic() {
        // Test the logic of inspect_tar_contents with simulated data
        // This tests the core logic without requiring actual tar files

        // Simulate a HashMap that would be returned by inspect_tar_contents
        // for an archive with a single directory containing files
        let mut components = std::collections::HashMap::new();
        components.insert("mydir".to_string(), true); // Directory with nested files

        let result: Vec<(String, bool)> = components.into_iter().collect();

        // Should have exactly one entry that is a directory
        assert_eq!(result.len(), 1);
        let (name, is_directory) = &result[0];
        assert_eq!(name, "mydir");
        assert!(*is_directory);

        // Test the should_strip_components logic with this result
        // This simulates what would happen if inspect_tar_contents returned this
        let should_strip = result.len() == 1 && result[0].1;
        assert!(should_strip);
    }

    #[test]
    fn test_inspect_tar_contents_curdir_prefix() {
        // Test that archives with "./" prefixed paths are handled correctly
        // This reproduces the bug from https://github.com/jdx/mise/discussions/7862
        use flate2::Compression;
        use flate2::write::GzEncoder;
        use tar::Builder;
        use tempfile::NamedTempFile;

        // Create a temp tar.gz with "./" prefixed paths (like unison's archive)
        let temp_file = NamedTempFile::new().unwrap();
        let gz = GzEncoder::new(temp_file.as_file(), Compression::default());
        let mut builder = Builder::new(gz);

        // Add entries with "./" prefix - simulating archive structure like:
        // ./dir1/file1
        // ./dir2/file2
        // ./standalone
        let mut header = tar::Header::new_gnu();
        header.set_size(0);
        header.set_mode(0o755);
        header.set_entry_type(tar::EntryType::Regular);
        header.set_cksum();

        // Add ./dir1/file1
        builder
            .append_data(&mut header.clone(), "./dir1/file1", std::io::empty())
            .unwrap();

        // Add ./dir2/file2
        builder
            .append_data(&mut header.clone(), "./dir2/file2", std::io::empty())
            .unwrap();

        // Add ./standalone (file at root with ./ prefix)
        builder
            .append_data(&mut header.clone(), "./standalone", std::io::empty())
            .unwrap();

        let gz = builder.into_inner().unwrap();
        gz.finish().unwrap();

        // Now test inspect_tar_contents
        let result = inspect_tar_contents(temp_file.path(), TarFormat::TarGz).unwrap();

        // Should have 3 top-level entries: dir1, dir2, standalone
        // NOT a single "." entry
        assert_eq!(
            result.len(),
            3,
            "Expected 3 top-level entries, got: {:?}",
            result
        );

        let names: std::collections::HashSet<_> = result.iter().map(|(n, _)| n.as_str()).collect();
        assert!(names.contains("dir1"), "Should contain dir1");
        assert!(names.contains("dir2"), "Should contain dir2");
        assert!(names.contains("standalone"), "Should contain standalone");
        assert!(!names.contains("."), "Should NOT contain '.' (CurDir)");

        // dir1 and dir2 should be marked as directories (have nested content)
        for (name, is_dir) in &result {
            if name == "dir1" || name == "dir2" {
                assert!(*is_dir, "{} should be marked as directory", name);
            } else if name == "standalone" {
                assert!(!*is_dir, "standalone should NOT be marked as directory");
            }
        }

        // Verify should_strip_components returns false (multiple top-level entries)
        let should_strip = should_strip_components(temp_file.path(), TarFormat::TarGz).unwrap();
        assert!(
            !should_strip,
            "Should NOT strip components for multi-entry archive"
        );
    }

    #[test]
    fn test_all_dirs_no_ceiling() {
        let start_dir = Path::new("/a/b/c");
        let ceiling_dirs = HashSet::new();

        let result = all_dirs(start_dir, &ceiling_dirs).unwrap();

        assert_eq!(result.len(), 4);
        assert!(result.contains(&PathBuf::from("/a/b/c")));
        assert!(result.contains(&PathBuf::from("/a/b")));
        assert!(result.contains(&PathBuf::from("/a")));
        assert!(result.contains(&PathBuf::from("/")));
    }

    #[test]
    fn test_all_dirs_with_ceiling() {
        let start_dir = Path::new("/a/b/c");
        let mut ceiling_dirs = HashSet::new();
        ceiling_dirs.insert(PathBuf::from("/a"));

        let result = all_dirs(start_dir, &ceiling_dirs).unwrap();

        assert_eq!(result.len(), 2);
        assert!(result.contains(&PathBuf::from("/a/b/c")));
        assert!(result.contains(&PathBuf::from("/a/b")));
        assert!(!result.contains(&PathBuf::from("/a")));
        assert!(!result.contains(&PathBuf::from("/")));
    }

    #[test]
    fn test_all_dirs_with_ceiling_at_start() {
        let start_dir = Path::new("/a/b/c");
        let mut ceiling_dirs = HashSet::new();
        ceiling_dirs.insert(PathBuf::from("/a/b/c"));

        let result = all_dirs(start_dir, &ceiling_dirs).unwrap();

        assert_eq!(result.len(), 0);
    }

    #[test]
    fn test_all_dirs_with_multiple_ceilings() {
        let start_dir = Path::new("/a/b/c/d/e");
        let mut ceiling_dirs = HashSet::new();
        ceiling_dirs.insert(PathBuf::from("/a/b"));
        ceiling_dirs.insert(PathBuf::from("/a/b/c/d"));

        let result = all_dirs(start_dir, &ceiling_dirs).unwrap();

        assert_eq!(result.len(), 1);
        assert!(result.contains(&PathBuf::from("/a/b/c/d/e")));
    }

    #[test]
    fn test_all_dirs_with_relative_path() {
        let start_dir = Path::new("a/b/c");
        let ceiling_dirs = HashSet::new();

        let result = all_dirs(start_dir, &ceiling_dirs).unwrap();

        assert!(result.contains(&PathBuf::from("a/b/c")));
        assert!(result.contains(&PathBuf::from("a/b")));
        assert!(result.contains(&PathBuf::from("a")));
    }

    #[test]
    fn test_tar_format_from_file_name() {
        assert_eq!(TarFormat::from_file_name("foo.tar.gz"), TarFormat::TarGz);
        assert_eq!(TarFormat::from_file_name("foo.tgz"), TarFormat::TarGz);
        assert_eq!(TarFormat::from_file_name("foo.tar.xz"), TarFormat::TarXz);
        assert_eq!(TarFormat::from_file_name("foo.txz"), TarFormat::TarXz);
        assert_eq!(TarFormat::from_file_name("foo.tar.bz2"), TarFormat::TarBz2);
        assert_eq!(TarFormat::from_file_name("foo.tbz2"), TarFormat::TarBz2);
        assert_eq!(TarFormat::from_file_name("foo.tar.zst"), TarFormat::TarZst);
        assert_eq!(TarFormat::from_file_name("foo.tzst"), TarFormat::TarZst);
        assert_eq!(TarFormat::from_file_name("foo.tar"), TarFormat::Tar);
        assert_eq!(TarFormat::from_file_name("foo.zip"), TarFormat::Zip);
        assert_eq!(TarFormat::from_file_name("foo.vsix"), TarFormat::Zip);
        assert_eq!(TarFormat::from_file_name("foo.7z"), TarFormat::SevenZip);
        assert_eq!(TarFormat::from_file_name("foo.gz"), TarFormat::Gz);
        assert_eq!(TarFormat::from_file_name("foo.xz"), TarFormat::Xz);
        assert_eq!(TarFormat::from_file_name("foo.bz2"), TarFormat::Bz2);
        assert_eq!(TarFormat::from_file_name("foo.zst"), TarFormat::Zst);
        assert_eq!(TarFormat::from_file_name("foo"), TarFormat::Raw);
        assert_eq!(TarFormat::from_file_name("foo.txt"), TarFormat::Raw);
    }

    #[test]
    fn test_untar_single_file() {
        use flate2::Compression;
        use flate2::write::GzEncoder;
        use std::io::Write;
        use tempfile::tempdir;

        let dir = tempdir().unwrap();
        let src_path = dir.path().join("test.gz");
        let dest_path = dir.path().join("test-out");

        // Create a dummy gzip file
        let file = File::create(&src_path).unwrap();
        let mut encoder = GzEncoder::new(file, Compression::default());
        encoder.write_all(b"hello world").unwrap();
        encoder.finish().unwrap();

        // untar (decompress) it
        untar(
            &src_path,
            &dest_path,
            &TarOptions {
                pr: None,
                ..TarOptions::new(TarFormat::Gz)
            },
        )
        .unwrap();

        // Verify output
        assert!(dest_path.exists());
        assert!(dest_path.is_file());
        let content = std::fs::read_to_string(&dest_path).unwrap();
        assert_eq!(content, "hello world");
    }

    #[test]
    fn test_untar_single_file_to_dir() {
        use flate2::Compression;
        use flate2::write::GzEncoder;
        use std::io::Write;
        use tempfile::tempdir;

        let dir = tempdir().unwrap();
        let src_path = dir.path().join("test_file.gz");
        let dest_dir = dir.path().join("out_dir");
        std::fs::create_dir(&dest_dir).unwrap();

        // Create a dummy gzip file
        let file = File::create(&src_path).unwrap();
        let mut encoder = GzEncoder::new(file, Compression::default());
        encoder.write_all(b"hello world").unwrap();
        encoder.finish().unwrap();

        // untar (decompress) it
        untar(
            &src_path,
            &dest_dir,
            &TarOptions {
                pr: None,
                ..TarOptions::new(TarFormat::Gz)
            },
        )
        .unwrap();

        // Verify output - should be out_dir/test_file
        let expected_path = dest_dir.join("test_file");
        assert!(expected_path.exists());
        assert!(expected_path.is_file());
        let content = std::fs::read_to_string(&expected_path).unwrap();
        assert_eq!(content, "hello world");
    }
}
